// root macro for simple limit calculations
//
//
#include <map>
#include <utility>
#include <iostream>

#include "TGraph.h"
#include "TGraphAsymmErrors.h"
#include "TCutG.h"
#include "TH1D.h"
#include "TH2F.h"
#include "TString.h"
#include "TFile.h"

// extra  utilities file
#include "LimitUtilities.h"
// #include "LimitUtilitiesTauTau.h"

// this is to define the cross section
// limits TGraphAsymmErrors if you don't have them in
// a single file
#include "LimitUtilitiesSetupLimits.h"

using namespace std;


bool DEBUG = false;


// TH2F which is mA,tanbeta versus xsec X BR
//
// Idea of the function: give in a TH2F the sigma X BR and compare at each point 
// with the limit, which is given in TGraphAsymmErrors format. Write output in a
// given root file.
void makeLimitPlot(TH2F *hin, TString tag, 
		   TGraphAsymmErrors *g_lim_obs,
		   TGraphAsymmErrors *g_lim_exp_1s,  TGraphAsymmErrors *g_lim_exp_2s,
		   TFile *foutput, TString folderName, TH2F *hfrac=0, bool useFrac=false
		   ) {
  // output that you want: 
  TGraphAsymmErrors *g_high   = new TGraphAsymmErrors();
  TGraphAsymmErrors *g_high_0 = new TGraphAsymmErrors();
  TGraphAsymmErrors *g_high_1 = new TGraphAsymmErrors();
  TGraphAsymmErrors *g_high_2 = new TGraphAsymmErrors();
  TCutG *gc_high   ;//= new TCutG();
  TCutG *gc_high_0 ;//= new TCutG();
  g_high->SetName("Observed");
  g_high_0->SetName("Expected");
  g_high_1->SetName("Expected1sigma");
  g_high_2->SetName("Expected2sigma");

  TGraphAsymmErrors *g_low   = new TGraphAsymmErrors();
  TGraphAsymmErrors *g_low_0 = new TGraphAsymmErrors();
  TGraphAsymmErrors *g_low_1 = new TGraphAsymmErrors();
  TGraphAsymmErrors *g_low_2 = new TGraphAsymmErrors();
  TCutG *gc_low   = new TCutG();
  TCutG *gc_low_0 = new TCutG();
  g_low->SetName("Observed");
  g_low_0->SetName("Expected");
  g_low_1->SetName("Expected1sigma");
  g_low_2->SetName("Expected2sigma");

  int counter_high_obs = 0;
  int counter_high_exp = 0;
  int counter_low_obs = 0;
  int counter_low_exp = 0;

  // for each point in the TGraphAsymmErrors do the interpolatio
  int N = g_lim_obs->GetN();
  //cout << "Will do for " << N << endl;
  for (int i=0; i < N; ++i) {
    // cout << "Looping: " << i  << endl;

    std::vector< TLimit_Info > v_info_obs;
    std::vector< TLimit_Info > v_info_exp;
    std::vector< TLimit_Info > v_info_p2s;
    std::vector< TLimit_Info > v_info_p1s;
    std::vector< TLimit_Info > v_info_m1s;
    std::vector< TLimit_Info > v_info_m2s;


    double mass, lim_obs, lim_exp, lim_p1s, lim_m1s, lim_p2s, lim_m2s;
    g_lim_obs->GetPoint(i, mass, lim_obs);

    TGraph g;
    getTGraph(hin, mass, g);
    TString name = tag + " " + TString::Format("%4.0f", mass);
    bool quiet=true;
    bool dodebug = false;
    // do not use the actual input limits, but just do the tautau lims
    if (useFrac) {
      //cout << "Will run interpolator for " << name << endl;
      TGraph gfrac, g1;
      getTGraph(hin, hfrac,  mass, g1, gfrac);
      //cout << "ready to run" << endl;
      Interpolator_TauTau(&g1, &gfrac, 99, name + " obs", v_info_obs, mass, false,dodebug);
      Interpolator_TauTau(&g1, &gfrac,  0, name + " exp", v_info_exp, mass, quiet,dodebug);
      Interpolator_TauTau(&g1, &gfrac,  2, name + " +2s", v_info_p2s, mass, quiet,dodebug);
      Interpolator_TauTau(&g1, &gfrac,  1, name + " +1s", v_info_p1s, mass, quiet,dodebug);
      Interpolator_TauTau(&g1, &gfrac, -1, name + " -1s", v_info_m1s, mass, quiet,dodebug);
      Interpolator_TauTau(&g1, &gfrac, -2, name + " -2s", v_info_m2s, mass, quiet,dodebug);
      //cout << "Interpolator run successfully" << endl;
    }
    else { // default method: use the actual input limits
      g_lim_exp_1s->GetPoint(i, mass, lim_exp);
      lim_p1s = lim_exp + g_lim_exp_1s->GetErrorYhigh(i);
      lim_m1s = lim_exp + g_lim_exp_1s->GetErrorYlow(i);
      lim_p2s = lim_exp + g_lim_exp_2s->GetErrorYhigh(i);
      lim_m2s = lim_exp + g_lim_exp_2s->GetErrorYlow(i);
      //dodebug = true;
      Interpolator(&g, lim_obs, name + " obs", v_info_obs, mass, false,dodebug);
      Interpolator(&g, lim_exp, name + " exp", v_info_exp, mass, quiet,dodebug);
      Interpolator(&g, lim_p2s, name + " +2s", v_info_p2s, mass, quiet,dodebug);
      Interpolator(&g, lim_p1s, name + " +1s", v_info_p1s, mass, quiet,dodebug);
      Interpolator(&g, lim_m1s, name + " -1s", v_info_m1s, mass, quiet,dodebug);
      Interpolator(&g, lim_m2s, name + " -2s", v_info_m2s, mass, quiet,dodebug);
    }


    int ncur=0;
    // fill in the TGraphAsymmErrors
    //cout << "Fill in the observed" << endl;
    /////////////////////////////////////////////////////////////
    ncur = v_info_obs.size();
    double ma_prev = 0; // this is in order to keep the previous mass value; if you see a mass twice, keep the second value
    for (int i=0; i < ncur; ++i) {
      TLimit_Info info = v_info_obs[i];
      //cout << "DEBUG: " << i << ", " << info.upcross << info.extrap << " mA=" << info.mA << " tb=" << info.tanb << endl;
      // see the last that is 11 or 10 and fill the high histo
      if ((i == ncur - 1)  && (info.upcross == 1) && (info.extrap == 1 || info.extrap == 0)) {
	g_high->SetPoint(counter_high_obs, info.mA, info.tanb);
	++counter_high_obs;
      }
      if (( (info.upcross == 0) && (info.extrap == 1) ) ||  (  (info.upcross == 0) && (info.extrap == 0) ) ) {
	if (ma_prev == info.mA) {
	  g_low->SetPoint(counter_low_obs-1, info.mA, info.tanb);
	} else {
	  g_low->SetPoint(counter_low_obs, info.mA, info.tanb);
	  ++counter_low_obs;	
	}
	ma_prev = info.mA;
      }
    }
    /////////////////////////////////////////////////////////////
    ncur = v_info_exp.size(); 

    // here you assume that all the bands exist!
    bool found_discrepancy = false;
    if ( ncur != int(v_info_p2s.size())  || ncur != int(v_info_p1s.size()) 
	 || ncur != int(v_info_m1s.size()) || ncur != int(v_info_m2s.size()) ) {
      cout << "WARNING!!! expected has: " << ncur << " entries and the bands: " << endl;
      cout << "+2 sigma:  " << v_info_p2s.size();
      cout << "  +1 sigma:  " << v_info_p1s.size();
      cout << "  -1 sigma:  " << v_info_m1s.size();
      cout << "  -2 sigma:  " << v_info_m2s.size() << endl;
      found_discrepancy = true;
    }
    ma_prev = 0;
    for (int i=0; i < ncur; ++i) {
      TLimit_Info info = v_info_exp[i];
      // if there is a expected limit, then there is also the band
      TLimit_Info info_p2s; //= v_info_p2s[i];
      TLimit_Info info_p1s; //= v_info_p1s[i];
      TLimit_Info info_m1s; //= v_info_m1s[i];
      TLimit_Info info_m2s; //= v_info_m2s[i];
      if (!found_discrepancy) {
	info_p2s = v_info_p2s[i];
	info_p1s = v_info_p1s[i];
	info_m1s = v_info_m1s[i];
	info_m2s = v_info_m2s[i];
      }
      //cout << "debug: " << i << " : " << info.upcross << " " << info.extrap << " :  " << counter_high_exp << endl;
      // see the last that is 11 or 10 and fill the high histo

      if ((i == ncur - 1)  && (info.upcross == 1) && (info.extrap == 1 || info.extrap == 0)) {
	g_high_0->SetPoint(counter_high_exp, info.mA, info.tanb);
	g_high_1->SetPoint(counter_high_exp, info.mA, info.tanb);
	g_high_2->SetPoint(counter_high_exp, info.mA, info.tanb);
	if (!found_discrepancy) {
	  g_high_1->SetPointError(counter_high_exp, 0.,0., 
				  info.tanb - info_m1s.tanb,info_p1s.tanb -info.tanb);
	  g_high_2->SetPointError(counter_high_exp, 0.,0., 
				  info.tanb - info_m2s.tanb,info_p2s.tanb -info.tanb);
	}
	++counter_high_exp;
      }
      if ( ( (info.upcross == 0) && (info.extrap == 1) )  || ( (info.upcross == 0) && (info.extrap == 0) )  ) {
	if (ma_prev == info.mA) {
	  g_low_0->SetPoint(counter_low_exp-1, info.mA, info.tanb);
	  g_low_1->SetPoint(counter_low_exp-1, info.mA, info.tanb);
	  g_low_2->SetPoint(counter_low_exp-1, info.mA, info.tanb);
	  if (!found_discrepancy) {
	    g_low_1->SetPointError(counter_low_exp-1, 0.,0., 
				   info.tanb - info_m1s.tanb,info_p1s.tanb -info.tanb);
	    g_low_2->SetPointError(counter_low_exp-1, 0.,0., 
				   info.tanb - info_m2s.tanb,info_p2s.tanb -info.tanb);
	  }
	  
	} else {
	  g_low_0->SetPoint(counter_low_exp, info.mA, info.tanb);
	  g_low_1->SetPoint(counter_low_exp, info.mA, info.tanb);
	  g_low_2->SetPoint(counter_low_exp, info.mA, info.tanb);
	  if (!found_discrepancy) {
	    g_low_1->SetPointError(counter_low_exp, 0.,0., 
				   info.tanb - info_m1s.tanb,info_p1s.tanb -info.tanb);
	    g_low_2->SetPointError(counter_low_exp, 0.,0., 
				   info.tanb - info_m2s.tanb,info_p2s.tanb -info.tanb);
	  }
	  ++counter_low_exp;	
	}
	ma_prev = info.mA;
      }
    }
    //cout << "finished with filling in limits" << endl;
  }
  //cout << "DEBUG: obs points " << counter_high_obs << ", " << counter_low_obs << endl;
  //cout << "DEBUG: exp points " << counter_high_exp << ", " << counter_low_exp << endl;

  // now make the TGCut plots
  gc_high  = new TCutG("gc_high", counter_high_obs+3);
  gc_high_0  = new TCutG("gc_high_0", counter_high_exp+3);

  gc_low  = new TCutG("gc_low", counter_low_obs+3);
  gc_low_0  = new TCutG("gc_low_0", counter_low_exp+3);

  gc_high->SetName("Observed_CutG");
  gc_high_0->SetName("Expected_CutG");
  gc_low->SetName("Observed_CutG");
  gc_low_0->SetName("Expected_CutG");

  double m_init, dump, m_init_0;
  //////////////////////////////////////////////////////
  g_high->GetPoint(0, m_init, dump);
  gc_high->SetPoint(0, m_init, 100.);  
  m_init_0 = m_init;
  for (int i=0; i < g_high->GetN(); ++i) {
    g_high->GetPoint(i, m_init, dump);
    gc_high->SetPoint(i+1, m_init, dump);
  }
  gc_high->SetPoint(counter_high_obs+1, m_init, 100.);
  gc_high->SetPoint(counter_high_obs+2, m_init_0, 100.);
  //  --> this is not correct in the edges, need to do one more interpolation
  g_low->GetPoint(0, m_init, dump);
  gc_high->SetPoint(0, m_init, 0.);  
  m_init_0 = m_init;
  for (int i=0; i < g_low->GetN(); ++i) {
    g_low->GetPoint(i, m_init, dump);
    gc_low->SetPoint(i+1, m_init, dump);
  }
  gc_low->SetPoint(counter_low_obs+1, m_init, 0.);
  gc_low->SetPoint(counter_low_obs+2, m_init_0, 0.);
  //////////////////////////////////////////////////////
  g_high_0->GetPoint(0, m_init, dump);
  gc_high_0->SetPoint(0, m_init, 100.); 
  m_init_0 = m_init;
  for (int i=0; i < g_high_0->GetN(); ++i) {
    g_high_0->GetPoint(i, m_init, dump);
    gc_high_0->SetPoint(i+1, m_init, dump);
  }
  gc_high_0->SetPoint(counter_high_exp+1, m_init, 100.);
  gc_high_0->SetPoint(counter_high_exp+2, m_init_0, 100.);
  // ................................................
  g_low_0->GetPoint(0, m_init, dump);
  gc_high_0->SetPoint(0, m_init, 0.);  
  m_init_0 = m_init;
  for (int i=0; i < g_low_0->GetN(); ++i) {
    g_low_0->GetPoint(i, m_init, dump);
    gc_low_0->SetPoint(i+1, m_init, dump);
  }
  gc_low_0->SetPoint(counter_low_obs+1, m_init, 0.);
  gc_low_0->SetPoint(counter_low_obs+2, m_init_0, 0.);
  //////////////////////////////////////////////////////

  gc_low_0->SetFillColor(1);
  gc_low_0->SetFillStyle(3004);

  gc_high_0->SetFillColor(1);
  gc_high_0->SetFillStyle(3004);



  // second part: write things in the output file
  foutput->mkdir(folderName+"_high");
  foutput->mkdir(folderName+"_low");
  foutput->cd(folderName+"_high");
  g_high->Write("Observed");
  g_high_0->Write("Expected");
  g_high_1->Write("Expected1sigma"); 
  g_high_2->Write("Expected2sigma"); 
  gc_high->Write("Observed_CutG"); 
  gc_high_0->Write("Expected_CutG");

  foutput->cd("../");
  foutput->cd(folderName+"_low");
  g_low->Write("Observed");
  g_low_0->Write("Expected");
  g_low_1->Write("Expected1sigma"); 
  g_low_2->Write("Expected2sigma"); 
  gc_low->Write("Observed_CutG"); 
  gc_low_0->Write("Expected_CutG");


  foutput->cd("../");


} // end function: makeLimitPlot( ... )


//
// this is to address the fact that you need an mA-tanbeta limit, but you
// have to constrain a different mass, e.g. mH;  
// The method also scans all the available mass points in the hMSSM ntuple grid
// and compares with the linearly interpolated limit from the TGraph.
// This may avoid the artificial behaviour of the limit around thresholds
// hin   : the sigmaxBR for mA--tanbeta
// hmass : the mass for mA--tanbeta 
void makeLimitPlotMass(TH2F *hin, TH2F *hmass, TString tag, 
		       TGraphAsymmErrors *g_lim_obs, 
		       TGraphAsymmErrors *g_lim_exp_1s,  TGraphAsymmErrors *g_lim_exp_2s,
		       TFile *foutput, TString folderName, double upper_mass=1000., 
		       TH2F *hfrac=0, bool useFrac=false
		       ) {
  // some preparation
  TGraphAsymmErrors *g_lim_exp = new TGraphAsymmErrors(*g_lim_exp_1s);

  TGraphAsymmErrors *g_lim_exp_p1s = new TGraphAsymmErrors(*g_lim_exp_1s);
  TGraphAsymmErrors *g_lim_exp_m1s = new TGraphAsymmErrors(*g_lim_exp_1s);
  for (int i=0; i<g_lim_exp_1s->GetN(); ++i) {
    double x, y, ymin, ymax;
    g_lim_exp_1s->GetPoint(i, x, y);
    ymin = g_lim_exp_1s->GetErrorYlow(i);
    ymax = g_lim_exp_1s->GetErrorYhigh(i);
    g_lim_exp_p1s->SetPoint(i, x, y+ymax);
    g_lim_exp_m1s->SetPoint(i, x, y-ymin);
  }

  TGraphAsymmErrors *g_lim_exp_p2s = new TGraphAsymmErrors(*g_lim_exp_2s);
  TGraphAsymmErrors *g_lim_exp_m2s = new TGraphAsymmErrors(*g_lim_exp_2s);
  for (int i=0; i<g_lim_exp_2s->GetN(); ++i) {
    double x, y, ymin, ymax;
    g_lim_exp_2s->GetPoint(i, x, y);
    ymin = g_lim_exp_2s->GetErrorYlow(i);
    ymax = g_lim_exp_2s->GetErrorYhigh(i);
    g_lim_exp_p2s->SetPoint(i, x, y+ymax);
    g_lim_exp_m2s->SetPoint(i, x, y-ymin);
  }

  // output that you want: 
  TGraphAsymmErrors *g_high   = new TGraphAsymmErrors();
  TGraphAsymmErrors *g_high_0 = new TGraphAsymmErrors();
  TGraphAsymmErrors *g_high_1 = new TGraphAsymmErrors();
  TGraphAsymmErrors *g_high_2 = new TGraphAsymmErrors();
  TCutG *gc_high   ;//= new TCutG();
  TCutG *gc_high_0 ;//= new TCutG();
  g_high->SetName("Observed");
  g_high_0->SetName("Expected");
  g_high_1->SetName("Expected1sigma");
  g_high_2->SetName("Expected2sigma");

  TGraphAsymmErrors *g_low   = new TGraphAsymmErrors();
  TGraphAsymmErrors *g_low_0 = new TGraphAsymmErrors();
  TGraphAsymmErrors *g_low_1 = new TGraphAsymmErrors();
  TGraphAsymmErrors *g_low_2 = new TGraphAsymmErrors();
  TCutG *gc_low   = new TCutG();
  TCutG *gc_low_0 = new TCutG();
  g_low->SetName("Observed");
  g_low_0->SetName("Expected");
  g_low_1->SetName("Expected1sigma");
  g_low_2->SetName("Expected2sigma");

  int counter_high_obs = 0;
  int counter_high_exp = 0;
  int counter_low_obs = 0;
  int counter_low_exp = 0;

  // for each point calculate the limit
  int N = hin->GetXaxis()->GetNbins();

  int n_limit = g_lim_obs->GetN();
  double mass_low, mass_high, lim; 
  g_lim_obs->GetPoint(0, mass_low, lim);
  g_lim_obs->GetPoint(n_limit-1, mass_high, lim);
  
  for (int i=1; i<=N; ++i) {
    double mass = hin->GetXaxis()->GetBinCenter(i);
    if (mass < 200 || mass > upper_mass) continue;
    //if (tag.Contains("A->Zh") && mass < 220.) continue;
    //if (mass < mass_low || mass > mass_high) continue;

    if (tag.Contains("H->WW") && mass < 270.) continue;

    TGraph g_XsecBR;
    TGraph g_mass;
    double tb_low = 0.7;

    if (tag.Contains("H->ZZ") && mass==240) tb_low = 1.0;
    getTGraph(hin, hmass,  mass, g_XsecBR, g_mass, tb_low);
    
    std::vector< TLimit_Info > v_info_obs;
    std::vector< TLimit_Info > v_info_exp;
    std::vector< TLimit_Info > v_info_p2s;
    std::vector< TLimit_Info > v_info_p1s;
    std::vector< TLimit_Info > v_info_m1s;
    std::vector< TLimit_Info > v_info_m2s;
  
    double lim_mass, lim_obs, lim_exp, lim_p1s, lim_m1s, lim_p2s, lim_m2s;

    TString name = tag + " " + TString::Format("%4.0f", mass);
    bool quiet=true;
    bool dodebug = false;
    if (useFrac) {
      // here you will assume mH = mA
      TGraph gfrac, g_XsecBR1;
      getTGraph(hin, hfrac,  mass, g_XsecBR1, gfrac);
      Interpolator_TauTau(&g_XsecBR1, &gfrac, 99, name + " obs", v_info_obs, mass, false,dodebug);
      Interpolator_TauTau(&g_XsecBR1, &gfrac,  0, name + " exp", v_info_exp, mass, quiet,dodebug);
      Interpolator_TauTau(&g_XsecBR1, &gfrac,  2, name + " +2s", v_info_p2s, mass, quiet,dodebug);
      Interpolator_TauTau(&g_XsecBR1, &gfrac,  1, name + " +1s", v_info_p1s, mass, quiet,dodebug);
      Interpolator_TauTau(&g_XsecBR1, &gfrac, -1, name + " -1s", v_info_m1s, mass, quiet,dodebug);
      Interpolator_TauTau(&g_XsecBR1, &gfrac, -2, name + " -2s", v_info_m2s, mass, quiet,dodebug);
    } else {
      bool forceM11 = false;
      dodebug = true;
      // if (tag.Contains("H->ZZ") && (mass == 350)) {forceM11 = true;}
      // if (tag.Contains("H->WW") && (mass == 280)) {forceM11 = true;}
      Interpolator(&g_XsecBR, &g_mass, g_lim_obs,name + " obs", v_info_obs, mass, false,dodebug, forceM11);

      // forceM11 = false;
      // if (tag.Contains("H->ZZ") && (mass == 240)) {forceM11 = true;}
      // if (tag.Contains("H->WW") && (mass == 280)) {forceM11 = true;}
      Interpolator(&g_XsecBR, &g_mass, g_lim_exp,name + " exp", v_info_exp, mass, false,dodebug, forceM11);
      // dodebug = false;
      // forceM11 = false;
      Interpolator(&g_XsecBR, &g_mass, g_lim_exp_p2s,name + " +2s", v_info_p2s, mass, quiet,dodebug, forceM11);
      Interpolator(&g_XsecBR, &g_mass, g_lim_exp_p1s,name + " +1s", v_info_p1s, mass, quiet,dodebug, forceM11);
      Interpolator(&g_XsecBR, &g_mass, g_lim_exp_m1s,name + " -1s", v_info_p1s, mass, quiet,dodebug, forceM11);
      Interpolator(&g_XsecBR, &g_mass, g_lim_exp_m2s,name + " -2s", v_info_p1s, mass, quiet,dodebug, forceM11);


      if (tag.Contains("A->Zh") && mass < 220. && v_info_obs.size() > 0) {      
	TLimit_Info info_obs;
	setupTLimit_Info(220., v_info_obs[0].tanb, v_info_obs[0].upcross, v_info_obs[0].extrap, info_obs);
	v_info_obs.clear();
	v_info_obs.push_back(info_obs);
      }
      if (tag.Contains("A->Zh") && mass < 220. && v_info_exp.size() > 0) {      
	TLimit_Info info_exp;
	setupTLimit_Info(220., v_info_exp[0].tanb, v_info_exp[0].upcross, v_info_exp[0].extrap, info_exp);
	v_info_exp.clear();
	v_info_exp.push_back(info_exp);
      }
       

    }

    int ncur=0;
    // fill in the TGraphAsymmErrors
    //cout << "Fill in the observed" << endl;
    /////////////////////////////////////////////////////////////
    ncur = v_info_obs.size();
    double ma_prev = 0; // this is in order to keep the previous mass value; if you see a mass twice, keep the second value
    for (int i=0; i < ncur; ++i) {
      TLimit_Info info = v_info_obs[i];
      //cout << "DEBUG: " << i << ", " << info.upcross << info.extrap << " mA=" << info.mA << " tb=" << info.tanb << endl;
      // see the last that is 11 or 10 and fill the high histo
      if ((i == ncur - 1)  && (info.upcross == 1) && (info.extrap == 1 || info.extrap == 0)) {
	g_high->SetPoint(counter_high_obs, info.mA, info.tanb);
	++counter_high_obs;
      }
      if (( (info.upcross == 0) && (info.extrap == 1) ) ||  (  (info.upcross == 0) && (info.extrap == 0) ) ) {
	if (ma_prev == info.mA) {
	  g_low->SetPoint(counter_low_obs-1, info.mA, info.tanb);
	} else {
	  g_low->SetPoint(counter_low_obs, info.mA, info.tanb);
	  ++counter_low_obs;	
	}
	ma_prev = info.mA;
      }
    }
    /////////////////////////////////////////////////////////////
    ncur = v_info_exp.size(); 

    // here you assume that all the bands exist!
    bool found_discrepancy = false;
    if ( ncur != int(v_info_p2s.size())  || ncur != int(v_info_p1s.size()) 
	 || ncur != int(v_info_m1s.size()) || ncur != int(v_info_m2s.size()) ) {
      // cout << "WARNING!!! expected has: " << ncur << " entries and the bands: " << endl;
      // cout << "+2 sigma:  " << v_info_p2s.size();
      // cout << "  +1 sigma:  " << v_info_p1s.size();
      // cout << "  -1 sigma:  " << v_info_m1s.size();
      // cout << "  -2 sigma:  " << v_info_m2s.size() << endl;
      found_discrepancy = true;
    }
    ma_prev = 0;
    for (int i=0; i < ncur; ++i) {
      TLimit_Info info = v_info_exp[i];
      // if there is a expected limit, then there is also the band
      TLimit_Info info_p2s; //= v_info_p2s[i];
      TLimit_Info info_p1s; //= v_info_p1s[i];
      TLimit_Info info_m1s; //= v_info_m1s[i];
      TLimit_Info info_m2s; //= v_info_m2s[i];
      if (!found_discrepancy) {
	info_p2s = v_info_p2s[i];
	info_p1s = v_info_p1s[i];
	info_m1s = v_info_m1s[i];
	info_m2s = v_info_m2s[i];
      }
      //cout << "debug: " << i << " : " << info.upcross << " " << info.extrap << " :  " << counter_high_exp << endl;
      // see the last that is 11 or 10 and fill the high histo

      if ((i == ncur - 1)  && (info.upcross == 1) && (info.extrap == 1 || info.extrap == 0)) {
	g_high_0->SetPoint(counter_high_exp, info.mA, info.tanb);
	g_high_1->SetPoint(counter_high_exp, info.mA, info.tanb);
	g_high_2->SetPoint(counter_high_exp, info.mA, info.tanb);
	if (!found_discrepancy) {
	  g_high_1->SetPointError(counter_high_exp, 0.,0., 
				  info.tanb - info_m1s.tanb,info_p1s.tanb -info.tanb);
	  g_high_2->SetPointError(counter_high_exp, 0.,0., 
				  info.tanb - info_m2s.tanb,info_p2s.tanb -info.tanb);
	}
	++counter_high_exp;
      }
      if ( ( (info.upcross == 0) && (info.extrap == 1) )  || ( (info.upcross == 0) && (info.extrap == 0) )  ) {
	if (ma_prev == info.mA) {
	  g_low_0->SetPoint(counter_low_exp-1, info.mA, info.tanb);
	  g_low_1->SetPoint(counter_low_exp-1, info.mA, info.tanb);
	  g_low_2->SetPoint(counter_low_exp-1, info.mA, info.tanb);
	  if (!found_discrepancy) {
	    g_low_1->SetPointError(counter_low_exp-1, 0.,0., 
				   info.tanb - info_m1s.tanb,info_p1s.tanb -info.tanb);
	    g_low_2->SetPointError(counter_low_exp-1, 0.,0., 
				   info.tanb - info_m2s.tanb,info_p2s.tanb -info.tanb);
	  }
	  
	} else {
	  g_low_0->SetPoint(counter_low_exp, info.mA, info.tanb);
	  g_low_1->SetPoint(counter_low_exp, info.mA, info.tanb);
	  g_low_2->SetPoint(counter_low_exp, info.mA, info.tanb);
	  if (!found_discrepancy) {
	    g_low_1->SetPointError(counter_low_exp, 0.,0., 
				   info.tanb - info_m1s.tanb,info_p1s.tanb -info.tanb);
	    g_low_2->SetPointError(counter_low_exp, 0.,0., 
				   info.tanb - info_m2s.tanb,info_p2s.tanb -info.tanb);
	  }
	  ++counter_low_exp;	
	}
	ma_prev = info.mA;
      }
    }
  }

  //cout << "DEBUG: obs points " << counter_high_obs << ", " << counter_low_obs << endl;
  //cout << "DEBUG: exp points " << counter_high_exp << ", " << counter_low_exp << endl;

  // now make the TGCut plots
  gc_high  = new TCutG("gc_high", counter_high_obs+3);
  gc_high_0  = new TCutG("gc_high_0", counter_high_exp+3);

  gc_low  = new TCutG("gc_low", counter_low_obs+3);
  gc_low_0  = new TCutG("gc_low_0", counter_low_exp+3);

  gc_high->SetName("Observed_CutG");
  gc_high_0->SetName("Expected_CutG");
  gc_low->SetName("Observed_CutG");
  gc_low_0->SetName("Expected_CutG");

  double m_init, dump, m_init_0;
  //////////////////////////////////////////////////////
  g_high->GetPoint(0, m_init, dump);
  gc_high->SetPoint(0, m_init, 100.);  
  m_init_0 = m_init;
  for (int i=0; i < g_high->GetN(); ++i) {
    g_high->GetPoint(i, m_init, dump);
    gc_high->SetPoint(i+1, m_init, dump);
  }
  gc_high->SetPoint(counter_high_obs+1, m_init, 100.);
  gc_high->SetPoint(counter_high_obs+2, m_init_0, 100.);
  //  --> this is not correct in the edges, need to do one more interpolation
  g_low->GetPoint(0, m_init, dump);
  //if (tag.Contains("gg->H->WW")) m_init = 270.;
  gc_high->SetPoint(0, m_init, 0.);  
  m_init_0 = m_init;
  for (int i=0; i < g_low->GetN(); ++i) {
    g_low->GetPoint(i, m_init, dump);
    gc_low->SetPoint(i+1, m_init, dump);
  }
  gc_low->SetPoint(counter_low_obs+1, m_init, 0.);
  gc_low->SetPoint(counter_low_obs+2, m_init_0, 0.);
  //////////////////////////////////////////////////////
  g_high_0->GetPoint(0, m_init, dump);
  gc_high_0->SetPoint(0, m_init, 100.); 
  m_init_0 = m_init;
  for (int i=0; i < g_high_0->GetN(); ++i) {
    g_high_0->GetPoint(i, m_init, dump);
    gc_high_0->SetPoint(i+1, m_init, dump);
  }
  gc_high_0->SetPoint(counter_high_exp+1, m_init, 100.);
  gc_high_0->SetPoint(counter_high_exp+2, m_init_0, 100.);
  // ................................................
  g_low_0->GetPoint(0, m_init, dump);
  gc_high_0->SetPoint(0, m_init, 0.);  
  m_init_0 = m_init;
  for (int i=0; i < g_low_0->GetN(); ++i) {
    g_low_0->GetPoint(i, m_init, dump);
    gc_low_0->SetPoint(i+1, m_init, dump);
  }
  gc_low_0->SetPoint(counter_low_obs+1, m_init, 0.);
  gc_low_0->SetPoint(counter_low_obs+2, m_init_0, 0.);
  //////////////////////////////////////////////////////

  gc_low_0->SetFillColor(1);
  gc_low_0->SetFillStyle(3004);

  gc_high_0->SetFillColor(1);
  gc_high_0->SetFillStyle(3004);



  // second part: write things in the output file
  foutput->mkdir(folderName+"_high");
  foutput->mkdir(folderName+"_low");
  foutput->cd(folderName+"_high");
  g_high->Write("Observed");
  g_high_0->Write("Expected");
  g_high_1->Write("Expected1sigma"); 
  g_high_2->Write("Expected2sigma"); 
  gc_high->Write("Observed_CutG"); 
  gc_high_0->Write("Expected_CutG");

  foutput->cd("../");
  foutput->cd(folderName+"_low");
  g_low->Write("Observed");
  g_low_0->Write("Expected");
  g_low_1->Write("Expected1sigma"); 
  g_low_2->Write("Expected2sigma"); 
  gc_low->Write("Observed_CutG"); 
  gc_low_0->Write("Expected_CutG");


  foutput->cd("../");


} // end function: makeLimitPlotMass( ... )



void test() {

  TString tfilename = "hMSSM_8TeV_1000_May18.root";

  TFile *f_xsecs = new TFile(tfilename,"read");

  TH2F *h_mh = (TH2F*) f_xsecs->Get("h_mh");
  TH2F *h_mH = (TH2F*) f_xsecs->Get("h_mH");
  TH2F *h_mA = (TH2F*) f_xsecs->Get("h_mh");
  getMAplot(h_mh, h_mA);

  // branching ratios from the file

  TH2F *h_brZZ_H = (TH2F*) f_xsecs->Get("h_brZZ_H");
  TH2F *h_brbb_h = (TH2F*) f_xsecs->Get("h_brbb_h");

  // production cross sections
  TH2F *h_ggF_xsec_A = (TH2F*) f_xsecs->Get("h_ggF_xsec_A");
  TH2F *h_ggF_xsec_H = (TH2F*) f_xsecs->Get("h_ggF_xsec_H");

  // gluon-fusion H->ZZ
  TH2F h_ggHZZ  = TH2F(*h_ggF_xsec_H);
  multiply_th2f(h_ggF_xsec_H,  h_brZZ_H, &h_ggHZZ);


  setupLimits_ZZ();

  TFile *foutput1 = new TFile("output1.root","recreate");
  
  cout << "ZZ limits test ---- MASS method:" << endl;
  makeLimitPlotMass(&h_ggHZZ, h_mH, "gg->H->ZZ", g_zz, g_zz_1, g_zz_2, foutput1, "HZZ", 500.);
  
  
  foutput1->Close();
  
  return;  

}
