#define USE_AS_MAIN
/* -*- mode: c++ -*- */
// Standard tutorial macro for performing an inverted  hypothesis test 
//
// This macro will perform a scan of tehe p-values for computing the limit
// 

#include "TFile.h"
#include "RooWorkspace.h"
#include "RooAbsPdf.h"
#include "RooRealVar.h"
#include "RooDataSet.h"
#include "RooStats/ModelConfig.h"
#include "RooRandom.h"
#include "TGraphErrors.h"
#include "TGraphAsymmErrors.h"
#include "TCanvas.h"
#include "TLine.h"
#include "TROOT.h"

#include "RooStats/AsymptoticCalculator.h"
#include "RooStats/HybridCalculator.h"
#include "RooStats/FrequentistCalculator.h"
#include "RooStats/ToyMCSampler.h"
#include "RooStats/HypoTestPlot.h"

#include "RooStats/NumEventsTestStat.h"
#include "RooStats/ProfileLikelihoodTestStat.h"
#include "RooStats/SimpleLikelihoodRatioTestStat.h"
#include "RooStats/RatioOfProfiledLikelihoodsTestStat.h"
#include "RooStats/MaxLikelihoodEstimateTestStat.h"

#include "RooStats/HypoTestInverter.h"
#include "RooStats/HypoTestInverterResult.h"
#include "RooStats/HypoTestInverterPlot.h"

#include <map>
using namespace RooFit;
using namespace RooStats;
using namespace std;

struct triplet {
  double mass;
  double tanbeta;
  TString fname;
  
};
TString RESULTSVERSION="cfg_new";
TString TAGMI="";
double TOLERANCE=1.0;
double confidenceLevel=0.95;
//
TString pname, wname, mname, bname, dname, fname, oname, sname; 
double xmass=125; // what else?
//
bool plotHypoTestResult = true;          // plot test statistic result at each point
bool writeResult = true;                 // write HypoTestInverterResult in a file 
TString resultFileName;                  // file with results (by default is built automatically using teh ws input file name)
bool optimize = true;                    // optmize evaluation of test statistic 
bool useVectorStore = true;              // convert data to use new roofit data store 
bool generateBinned = false;             // generate binned data sets 
bool noSystematics = false;              // force all systematics to be off (i.e. set all nuisance parameters as constat
                                         // to their nominal values)
double nToysRatio = 2;                   // ratio Ntoys S+b/ntoysB
double maxPOI = -1;                      // max value used of POI (in case of auto scan) 
bool useProof = false;                    // use Proof Light when using toys (for freq or hybrid)
int nworkers = 4;                        // number of worker for Proof
bool rebuild = false;                    // re-do extra toys for computing expected limits and rebuild test stat
                                         // distributions (N.B this requires much more CPU (factor is equivalent to nToyToRebuild)
int nToyToRebuild = 100;                 // number of toys used to rebuild 
int initialFit = -1;                     // do a first  fit to the model (-1 : default, 0 skip fit, 1 do always fit) 
int randomSeed = -1;                     // random seed (if = -1: use default value, if = 0 always random )
                                         // NOTE: Proof uses automatically a random seed

std::string massValue = "";              // extra string to tag output file of result 
std::string  minimizerType = "";                  // minimizer type (default is what is in ROOT::Math::MinimizerOptions::DefaultMinimizerType()
int   printLevel = 0;                    // print level for debugging PL test statistics and calculators  

// some extra from Nikos
double SCANMIN  = -1;
double SCANMAX  = -1;
double SCANSTEP = -1;

typedef  std::vector< triplet > filesHolder;

triplet make_triplet( TString fname, double mass, double tanbeta);

void printDebugTableMA(double mass, 
		       map<double, double> tb_observed,
		       map<double, double> tb_median,
		       map<double, double> tb_medianP2sigma,
		       map<double, double> tb_medianP1sigma,
		       map<double, double> tb_medianM1sigma,
		       map<double, double> tb_medianM2sigma, vector<double> tanBetaList
		       );
double getExrapolationGivenTanb(double given_tanb, map<double, double> myMap);
double getTanBeta(map<double, double> myMap);

void plotMaker(std::map<double, double> median,
	       std::map<double, double> medianP2sigma,
	       std::map<double, double> medianP1sigma,
	       std::map<double, double> medianM1sigma,
	       std::map<double, double> medianM2sigma,
	       std::map<double, double> observed,
	       bool onlyExpectation, TString pName = "testPlot.cc"
	       );
void runMSSMHtautauLimitsConfigGeneral(TString pathPref, TString WhatSyst, TString pName, double Mmin=0, double Mmax=999);
bool checkInfinity(TString name, double number);

double  FACTOR=1.2;
bool runMSSM(true);
TString IndPref("BB");
bool RUNTOYS(false);

// ------------------> functions <-----------------------------------------
triplet make_triplet( TString fname, double mass, double tanbeta) {
  triplet res;
  res.mass = mass;
  res.tanbeta = tanbeta;
  res.fname = fname;
  return res;
}

filesHolder makeFilesHolder(TString pathName, TString WhatSyst, double mass, double tanbeta) {
  TString smass(""); smass += int(mass);
  if (mass<100) smass="0"+smass;

//   vector<int> tanbetalist;
//   if (runMSSM) {

//     // --> tanbeta list to push back depends of the record
//     const    int N_list_5=17;
//     int list_5[N_list_5] = {5,8,10,13,16,20,23,26,30,35,40,45,50,55,60,65,70};
//     //                        1 2 3   4  5  6  7  8  9  0  1  2 3 4   5  6  7
//     const    int N_list_16=13;
//     int list_16[N_list_16] = {16,20,23,26,30,35,40,45,50,55,60,65,70};
//     //
//     const    int N_list_30=9;
//     int list_30[N_list_30] = {30,35,40,45,50,55,60,65,70};
//     //
//     const    int N_list_40=7;
//     int list_40[N_list_40] = {40,45,50,55,60,65,70};
//     // choose the list now
//     // --
//     int NList;
//     int *theList = list_5;
//     if (mass <301) {
//       NList=N_list_5;
//       theList = list_5;
//     }
//     else if (mass<401) {
//       NList=N_list_16;
//       theList = list_16;
//     }
//     else {
//       NList=N_list_30;
//       theList = list_30;
//     }

//     for (int i=0;i<NList; ++i){
//       tanbetalist.push_back(theList[i]);
//     }


//   }
//   else {
//     cout << "Warning! will run Model Independent limit" << endl;
//     tanbetalist.push_back(20);
//   }

//   filesHolder files;
//   int ntanbeta = tanbetalist.size();

//   for (int i=0; i<ntanbeta; ++i) {
//     TString s_tb(""); s_tb += tanbetalist[i];
//     if (tanbetalist[i] < 10) s_tb = "0"+s_tb;

//     files.push_back( make_triplet(pathName+"mA"+smass+"tb"+s_tb+"_comb_combined_"+WhatSyst+"_model.root",  mass, double( tanbetalist[i])  ));
//   }


  filesHolder files;
  TString s_tb(""); s_tb += tanbeta;
  if (tanbeta < 10) s_tb = "0"+s_tb;

  files.push_back( make_triplet(pathName+"mA"+smass+"tb"+s_tb+TAGMI+"_comb_combined_"+WhatSyst+"_model.root",  mass, tanbeta));
  // mssm_mA450tb30_comb_combined_AllSyst_model.root
  return files;
  

}

vector<filesHolder> makeVectorFilesHolder(TString pathName, TString WhatSyst, double massMin, double massMax) {

//   vector<filesHolder> files;
//   vector<double> masses;
//   masses.push_back( 90.);
//   masses.push_back(100.);
//   masses.push_back(110.);
//   masses.push_back(120.);
//   masses.push_back(130.);
//   masses.push_back(140.);
//   masses.push_back(150.);
//   masses.push_back(170.);
//   masses.push_back(200.);
//   masses.push_back(250.);
//   masses.push_back(300.);

//   masses.push_back(350.);
//   masses.push_back(400.);
//   masses.push_back(450.);
//   masses.push_back(500.);
//   masses.push_back(600.);

//   for (int i=0; i< masses.size(); ++i) {
//     if (masses[i]>=massMin and masses[i]<=massMax) {
//       printf("--> Will Add Mass Point %3.0f \n", masses[i]);
//       files.push_back( makeFilesHolder(pathName, WhatSyst,  masses[i]) );
//     }
//   }

  vector<filesHolder> files;

  files.push_back( makeFilesHolder(pathName, WhatSyst,  massMin, massMax) );

  

  return files;
}


bool checkInfinity(TString name, double number) {
  if (number > 999999) {
    cout << "WARNING! Infinity observed in " << name << endl;
    return true;
  }
  return false;
}

void plotMaker(std::map<double, double> map_median,
	       std::map<double, double> map_medianP2sigma,
	       std::map<double, double> map_medianP1sigma,
	       std::map<double, double> map_medianM1sigma,
	       std::map<double, double> map_medianM2sigma,
	       std::map<double, double> map_observed,
	       bool onlyExpectation, TString pName 
	       ) {
  ofstream myfile (pName.Data());

  myfile << " void test() {" << endl;
  myfile << "   gROOT->Reset();" << endl;
  myfile << "   gROOT->ProcessLine(\".L ~/atlasStyle.C\");" << endl;
  myfile << "   gROOT->ProcessLine(\"setAtlasStyle()\");" << endl;

  myfile << "   TGraphAsymmErrors *g  = new TGraphAsymmErrors(); // data observed" << endl;    
  myfile << "   TGraphAsymmErrors *g0 = new TGraphAsymmErrors(); // data expected" << endl;    
  myfile << "   TGraphAsymmErrors *g1 = new TGraphAsymmErrors(); // data expected: 1 sigma" << endl;
  myfile << "   TGraphAsymmErrors *g2 = new TGraphAsymmErrors(); // data expected: 2 sigma" << endl;

  std::map<double, double>::iterator it_map_median = map_median.begin();

  int i(-1);
  for (; it_map_median != map_median.end(); ++it_map_median) {
    ++i;


    double mass = (*it_map_median).first;
    double median =(*it_map_median).second;

    double medianP2sig = map_medianP2sigma[mass];
    double medianP1sig = map_medianP1sigma[mass];
    double medianM1sig = map_medianM1sigma[mass];
    double medianM2sig = map_medianM2sigma[mass];

    double observed    = 0;
    if (onlyExpectation == false) {
      observed    = map_observed[mass];
      if (observed != observed) observed = 0;
      myfile << "   g->SetPoint(" << i << ", "<< mass << ", " << observed  <<");"  << endl;
    }
    myfile <<   "   g0->SetPoint(" << i << ", " << mass << ", " << median  <<");"  << endl;
    myfile <<   "   g1->SetPoint(" << i << ", " << mass << ", " << median  <<");"  << endl;
    myfile <<   "   g2->SetPoint(" << i << ", " << mass << ", " << median  <<");"  << endl;

    myfile <<   "   g1->SetPointError(" << i << ", 0,0," << median-medianM1sig << ", "  
	 <<   medianP1sig-median  <<   ");"  << endl;
    myfile <<   "   g2->SetPointError(" << i << ", 0,0," << median-medianM2sig << ", "  
	 <<   medianP2sig-median  <<   ");"  << endl;
  }
  myfile << "   TCanvas *c = new TCanvas();" << endl;
  
  myfile << "   g->SetMarkerStyle(20);" << endl;

  if (runMSSM) {
    myfile << "   g0->GetYaxis()->SetTitle(\"tan#beta\");" << endl;
    myfile << "   g0->GetXaxis()->SetTitle(\"m_{A} [GeV/c^{2}]\");" << endl;
  }
  else {
    if (pName.Contains("bTagged")) {
      myfile << "   g0->GetYaxis()->SetTitle(\"95% CLs #sigma(bb#rightarrow#phi) #times BR(#phi#rightarrow#tau#tau#rightarrow lep had) [pb]\");" << endl;
    }
    else {
      myfile << "   g0->GetYaxis()->SetTitle(\"95% CLs #sigma(gg#rightarrow#phi) #times BR(#phi#rightarrow#tau#tau#rightarrow lep had) [pb]\");" << endl;
    }
    myfile << "   g0->GetXaxis()->SetTitle(\"m_{#phi} [GeV/c^{2}]\");" << endl;
    myfile << "   gPad->SetLogy();" << endl;
  }
  myfile << "   g0->SetLineStyle(2);" << endl;
  myfile << "   g1->SetLineStyle(2);" << endl;
  myfile << "   g2->SetLineStyle(2);" << endl;
  myfile << "   g0->SetLineWidth(2);" << endl;
  myfile << "   g1->SetLineWidth(2);" << endl;
  myfile << "   g2->SetLineWidth(2);" << endl;
  myfile << "   g1->SetFillColor(kYellow+1);" << endl;
  myfile << "   g2->SetFillColor(kYellow+2);" << endl;

  myfile << "   g0->GetYaxis()->SetRangeUser(0,70);" << endl;
  myfile << "   g0->Draw(\"AL\");" << endl;
  myfile << "   g2->Draw(\"E3,same\");" << endl;
  myfile << "   g1->Draw(\"E3,same\");" << endl;

  myfile << "   g0->Draw(\"L,same\");" << endl;


  myfile << "   //g->Draw(\"PL,same\");" << endl;



  myfile <<    "    TLegend *leg = new TLegend(0.60, 0.60, 0.90, 0.80);"  << endl;
  myfile <<    "    leg->SetFillColor(kWhite);" << endl;
  myfile <<    "    leg->SetFillStyle(0);" << endl;
  myfile <<    "    leg->SetLineColor(0);" << endl;

  myfile <<    "    g2->SetLineColor(0);" << endl;
  myfile <<    "    g1->SetLineColor(0);" << endl;
  myfile <<    "    g2->SetLineStyle(0);" << endl;
  myfile <<    "    g1->SetLineStyle(0);" << endl;

  myfile <<    "    leg->AddEntry(g0, \"Expected CLs\", \"L\");" << endl;
  myfile <<    "    leg->AddEntry(g1,\"1 #sigma\", \"F\");" << endl;
  myfile <<    "    leg->AddEntry(g2,\"2 #sigma\", \"F\");" << endl;
  myfile <<    "    leg->Draw(\"same\");" << endl;
  myfile <<    "    TLatex *l = new TLatex;" << endl;
  myfile <<    "    l->SetNDC();" << endl;
  myfile <<    "    l->SetTextFont(72);" << endl;
  myfile <<    "    l->SetTextColor(1);" << endl;
  myfile <<    "    l->SetTextSize(0.04);" << endl;
  myfile <<    "    l->DrawLatex(0.68,0.9,\"ATLAS Internal\");" << endl;

  myfile <<    "    TLatex *l1 = new TLatex;" << endl;
  myfile <<    "    l1->SetNDC();" << endl;
  myfile <<    "    l1->SetTextFont(72);" << endl;
  myfile <<    "    l1->SetTextColor(1);" << endl;
  myfile <<    "    l1->SetTextSize(0.04);" << endl;

  TString labelname("");
  if (pName.Contains("Case1")) labelname += "b-tag ";
  if (pName.Contains("Case3")) labelname += "b-veto ";
  if (pName.Contains("Case6")) labelname += "combined ";
  labelname += "sample";
  if (pName.Contains("StatOnly")) labelname += ", stat only";

  myfile <<    "    l1->DrawLatex(0.68,0.83,\""<< labelname.Data() << "\");" << endl;

  TString fname1("limit_");
  TString fname2("limit_");
  TString theFac(""); theFac += int(FACTOR*100);
  if (pName.Contains("exp_Case1_bTaggedDef_AllSyst"))  { fname1 += "bTag_allSyst"+theFac+".eps";  fname2 += "bTag_allSyst"+theFac+".png";}
  if (pName.Contains("exp_Case1_bTaggedDef_StatOnly")) { fname1 += "bTag_StatOnly"+theFac+".eps"; fname2 += "bTag_StatOnly"+theFac+".png";}

  if (pName.Contains("exp_Case3_Compl_StatOnly")) { fname1 += "bVeto_StatOnly"+theFac+".eps"; fname2 += "bVeto_StatOnly"+theFac+".png";}
  if (pName.Contains("exp_Case3_Compl_AllSyst"))  { fname1 += "bVeto_allSyst"+theFac+".eps";  fname2 += "bVeto_allSyst"+theFac+".png";}

  if (pName.Contains("exp_Case6_COMBbTagCom_StatOnly")){ fname1 += "comb_StatOnly"+theFac+".eps"; fname2 += "comb_StatOnly"+theFac+".png";}
  if (pName.Contains("exp_Case6_COMBbTagCom_AllSyst")) { fname1 += "comb_allSyst"+theFac+".eps";  fname2 += "comb_allSyst"+theFac+".png";}

  if (pName.Contains("exp_Case4_ComplMET_StatOnly")) { fname1 += "bVetoMET_StatOnly"+theFac+".eps"; fname2 += "bVetoMET_StatOnly"+theFac+".png";}
  if (pName.Contains("exp_Case4_ComplMET_AllSyst"))  { fname1 += "bVetoMET_allSyst"+theFac+".eps";  fname2 += "bVetoMET_allSyst"+theFac+".png";}

  if (pName.Contains("exp_Case2_bTaggedMET_AllSyst"))  { fname1 += "bTagMET_allSyst"+theFac+".eps";  fname2 += "bTagMET_allSyst"+theFac+".png";}
  if (pName.Contains("exp_Case2_bTaggedMET_StatOnly")) { fname1 += "bTagMET_StatOnly"+theFac+".eps"; fname2 += "bTagMET_StatOnly"+theFac+".png";}

  if (pName.Contains("exp_Case8_COMBbTagComMET_StatOnly")){ fname1 += "combMET_StatOnly"+theFac+".eps"; fname2 += "combMET_StatOnly"+theFac+".png";}
  if (pName.Contains("exp_Case8_COMBbTagComMET_AllSyst")) { fname1 += "combMET_allSyst"+theFac+".eps";  fname2 += "combMET_allSyst"+theFac+".png";}


  myfile <<    "    c->Print(\"" << fname1  << "\");" << endl;
  myfile <<    "    c->Print(\"" << fname2  << "\");" << endl;




 

  myfile << "}" << endl;

}

double getExrapolationGivenTanb(double given_tanb, map<double, double> myMap) {
  // find this tanb and make extrapolation/interpolation
  
  map<double, double>::iterator it = myMap.begin();

  // check the first entry
  double tanb_previous = it->first;
  double mu_previous   = it->second;

  map<double, double>::iterator it_previous = it;
  ++it;
  double tanb; // current values
  double mu;   // current values
  //
  if (given_tanb < tanb_previous) {
    tanb = it->first;
    mu = it->second;
    double alpha = (mu_previous - mu)/(tanb_previous-tanb);
    double beta  = mu_previous - alpha*tanb_previous;
    return alpha*given_tanb + beta; // mu for tanbeta = given tanb
  }
  //
  for (; it != myMap.end(); ++it) {
    tanb = it->first;
    mu = it->second;

    if (given_tanb < tanb and given_tanb > tanb_previous) {
      // make extrapolation now
//       cout << "DEBUG-EXTAP: Warning: using extrapolation" << endl;
//       cout << "ERR-DBG  tanb= " << tanb << ", " << tanb_previous << endl;
//       cout << "ERR-DBG  mu=   " << mu << ", " << mu_previous << endl;

      double alpha = (mu_previous - mu)/(tanb_previous-tanb);
      double beta  = mu_previous - alpha*tanb_previous;
      //cout << "ERR-BDG a: " << alpha << ", b: " << beta << endl; 
      return alpha*given_tanb + beta; // mu for tanbeta = given tanb
    }
    tanb_previous = it->first;
    mu_previous   = it->second;
    it_previous = it;
  }

  // if you have reached the end and you haven't found anything
  // you extrapolate
  tanb = tanb_previous;
  mu = mu_previous;
  --it;  --it;
  tanb_previous = it->first;
  mu_previous   = it->second;

//   cout << "DEBUG-EXTAP: Warning: using extrapolation" << endl;
//   cout << "ERR-DBG  tanb= " << tanb << ", " << tanb_previous << endl;
//   cout << "ERR-DBG  mu=   " << mu << ", " << mu_previous << endl;

  double alpha = (mu_previous - mu)/(tanb_previous-tanb);
  double beta  = mu_previous - alpha*tanb_previous;
  //cout << "ERR-BDG a: " << alpha << ", b: " << beta << endl; 
  return alpha*given_tanb + beta; // mu for tanbeta = given tanb



}

//
// this requires the tanbeta to increase
// as tanbeta increases then the observed / expected limit shoul decrease
//
double getTanBeta(map<double, double> myMap) {
  //
  // make linear interpolation
  // or linear extrapolation if the value is outside the limits
  map<double, double>::iterator it = myMap.begin();

  // check the first entry
  double tanb_previous = it->first;
  double mu_previous   = it->second;

  // if the first entry is 1 within 0.01 then return this value and issue a warning
  if (fabs(mu_previous - 1) < 0.01) {
    printf("DEBUG-EXTAP: Warning: first value at tanbeta=%3.0f has mu=%6.4f\n", tanb_previous, mu_previous);
    return tanb_previous;
  }

  map<double, double>::iterator it_previous = it;
  ++it;
  double tanb; // current values
  double mu;   // current values
  //
  if ( mu_previous < 1) {
    tanb = it->first;
    mu = it->second;
    double alpha = (tanb_previous-tanb)/(mu_previous - mu);
    double beta  = tanb_previous - alpha*mu_previous;
    return alpha/* *1.*/ + beta; // tanbeta for mu = 1
  }
  //
  for (; it != myMap.end(); ++it) {
    tanb = it->first;
    mu = it->second;

    if (mu_previous > 1 and mu < 1) {
      // make extrapolation
      cout << "ERR-DBG  tanb= " << tanb << ", " << tanb_previous << endl;
      cout << "ERR-DBG  mu=   " << mu << ", " << mu_previous << endl;

      double alpha = (tanb_previous-tanb)/(mu_previous - mu);
      double beta  = tanb_previous - alpha*mu_previous;
      cout << "ERR-BDG a: " << alpha << ", b: " << beta << endl; 
      return alpha/* *1.*/ + beta; // tanbeta for mu = 1
    }

    tanb_previous = it->first;
    mu_previous   = it->second;
    it_previous = it;
  }

  // if you have reached the end and you haven't found anything
  // you extrapolate
  tanb = tanb_previous;
  mu = mu_previous;
  --it;  --it;
  tanb_previous = it->first;
  mu_previous   = it->second;

  cout << "DEBUG-EXTAP: Warning: using extrapolation" << endl;
  cout << "ERR-DBG  tanb= " << tanb << ", " << tanb_previous << endl;
  cout << "ERR-DBG  mu=   " << mu << ", " << mu_previous << endl;

  double alpha = (tanb_previous-tanb)/(mu_previous - mu);
  double beta  = tanb_previous - alpha*mu_previous;
  cout << "ERR-BDG a: " << alpha << ", b: " << beta << endl; 
  return alpha/* *1.*/ + beta; // tanbeta for mu = 1    
}

void printDebugTableMA(double mass, 
		       map<double, double> tb_observed,
		       map<double, double> tb_median,
		       map<double, double> tb_medianP2sigma,
		       map<double, double> tb_medianP1sigma,
		       map<double, double> tb_medianM1sigma,
		       map<double, double> tb_medianM2sigma,
		       vector<double> tanBetaList
		       ) {

  vector<double>::iterator it = tanBetaList.begin();
  for (; it != tanBetaList.end(); ++it) {
    double tanb = (*it);

    double obs    = 0.;
    double med    = 0.;
    double medP2  = 0.;
    double medP1  = 0.;
    double medM1  = 0.;
    double medM2  = 0.;

    map<double, double>::iterator it_obs   = tb_observed.find(tanb);
    map<double, double>::iterator it_med   = tb_median.find(tanb);
    map<double, double>::iterator it_medP2 = tb_medianP2sigma.find(tanb);
    map<double, double>::iterator it_medP1 = tb_medianP1sigma.find(tanb);
    map<double, double>::iterator it_medM1 = tb_medianM1sigma.find(tanb);
    map<double, double>::iterator it_medM2 = tb_medianM2sigma.find(tanb);

    if (it_obs != tb_observed.end())  obs = (it_obs->second);
    if (it_med != tb_median.end())    med = (it_med->second);

    if (it_medP2 != tb_medianP2sigma.end())   medP2 = (it_medP2->second);
    if (it_medP1 != tb_medianP1sigma.end())   medP1 = (it_medP1->second);
    if (it_medM1 != tb_medianM1sigma.end())   medM1 = (it_medM1->second);
    if (it_medM2 != tb_medianM2sigma.end())   medM2 = (it_medM2->second);

    TString error_mes(" ");
    if (SCANMIN > med or SCANMAX < med) error_mes += "SCANERROR-Median ";
    if (SCANMIN > obs or SCANMAX < obs) error_mes += "SCANERROR-Observ ";
    printf("DEBUG-TABLE Scan parameters: step: %5.3f  min: %8.3f max: %8.3f %s \n ", SCANSTEP, SCANMIN, SCANMAX, error_mes.Data());
    //  printf("DEBUG-TABLE  Summary for mA %4.0f\n",mass);
    printf("DEBUG-TABLE  Tanb     Observed Median  +2sig  +1sig  -1sig  -2sig\n");
    //      DEBUG-TABLE  tanb=   5    0.00   2.31   4.32   3.22   1.67   1.24

    printf("DEBUG-TABLE mA= %4.0f tanb= %3.0f  %6.2f %6.2f %6.2f %6.2f %6.2f %6.2f\n",
	   mass, tanb, obs, med, medP2, medP1, medM1, medM2 );

    
  }

//   map<double, double>::iterator it = tb_observed.begin();
//   for(; it != tb_observed.end(); ++it) {
//     double tanb = it->first;
//     printf("DEBUG-TABLE  tanb= %3.0f  %6.2f %6.2f %6.2f %6.2f %6.2f %6.2f\n",
// 	   tanb, tb_observed[tanb], tb_median[tanb],
// 	   tb_medianP2sigma[tanb], tb_medianP1sigma[tanb], 
// 	   tb_medianM1sigma[tanb], tb_medianM2sigma[tanb]);
//   }

}




// ---------------------------------------------------------------


// internal class to run the inverter and more

namespace RooStats { 

   class HypoTestInvTool{

   public:
      HypoTestInvTool();
      ~HypoTestInvTool(){};

      HypoTestInverterResult * 
      RunInverter(RooWorkspace * w, 
                  const char * modelSBName, const char * modelBName, 
                  const char * dataName,
                  int type,  int testStatType, 
                  bool useCLs, 
                  int npoints, double poimin, double poimax, int ntoys, 
                  bool useNumberCounting = false, 
                  const char * nuisPriorName = 0);



      vector<double>
      AnalyzeResult( HypoTestInverterResult * r,
                     int calculatorType,
                     int testStatType, 
                     bool useCLs,  
                     int npoints,
                     const char * fileNameBase = 0 );

      void SetParameter(const char * name, const char * value);
      void SetParameter(const char * name, bool value);
      void SetParameter(const char * name, int value);
      void SetParameter(const char * name, double value);

   private:

      bool mPlotHypoTestResult;
      bool mWriteResult;
      bool mOptimize;
      bool mUseVectorStore;
      bool mGenerateBinned;
      bool mUseProof;
      bool mRebuild;
      int     mNWorkers;
      int     mNToyToRebuild;
      int     mPrintLevel;
      int     mInitialFit; 
      int     mRandomSeed; 
      double  mNToysRatio;
      double  mMaxPoi;
      std::string mMassValue;
      std::string mMinimizerType;                  // minimizer type (default is what is in ROOT::Math::MinimizerOptions::DefaultMinimizerType()
      TString     mResultFileName; 
   };

} // end namespace RooStats

RooStats::HypoTestInvTool::HypoTestInvTool() : mPlotHypoTestResult(true),
                                               mWriteResult(false),
                                               mOptimize(true),
                                               mUseVectorStore(true),
                                               mGenerateBinned(false),
                                               mUseProof(false),
                                               mRebuild(false),
                                               mNWorkers(4),
                                               mNToyToRebuild(100),
                                               mPrintLevel(0),
                                               mInitialFit(-1),
                                               mRandomSeed(-1),
                                               mNToysRatio(2),
                                               mMaxPoi(-1),
                                               mMassValue(""),
                                               mMinimizerType(""),
                                               mResultFileName() {
}



void
RooStats::HypoTestInvTool::SetParameter(const char * name, bool value){
   //
   // set boolean parameters
   //

   std::string s_name(name);

   if (s_name.find("PlotHypoTestResult") != std::string::npos) mPlotHypoTestResult = value;
   if (s_name.find("WriteResult") != std::string::npos) mWriteResult = value;
   if (s_name.find("Optimize") != std::string::npos) mOptimize = value;
   if (s_name.find("UseVectorStore") != std::string::npos) mUseVectorStore = value;
   if (s_name.find("GenerateBinned") != std::string::npos) mGenerateBinned = value;
   if (s_name.find("UseProof") != std::string::npos) mUseProof = value;
   if (s_name.find("Rebuild") != std::string::npos) mRebuild = value;

   return;
}



void
RooStats::HypoTestInvTool::SetParameter(const char * name, int value){
   //
   // set integer parameters
   //

   std::string s_name(name);

   if (s_name.find("NWorkers") != std::string::npos) mNWorkers = value;
   if (s_name.find("NToyToRebuild") != std::string::npos) mNToyToRebuild = value;
   if (s_name.find("PrintLevel") != std::string::npos) mPrintLevel = value;
   if (s_name.find("InitialFit") != std::string::npos) mInitialFit = value;
   if (s_name.find("RandomSeed") != std::string::npos) mRandomSeed = value;

   return;
}



void
RooStats::HypoTestInvTool::SetParameter(const char * name, double value){
   //
   // set double precision parameters
   //

   std::string s_name(name);

   if (s_name.find("NToysRatio") != std::string::npos) mNToysRatio = value;
   if (s_name.find("MaxPOI") != std::string::npos) mMaxPoi = value;

   return;
}



void
RooStats::HypoTestInvTool::SetParameter(const char * name, const char * value){
   //
   // set string parameters
   //

   std::string s_name(name);

   if (s_name.find("MassValue") != std::string::npos) mMassValue.assign(value);
   if (s_name.find("MinimizerType") != std::string::npos) mMinimizerType.assign(value);
   if (s_name.find("ResultFileName") != std::string::npos) mResultFileName = value;

   return;
}



vector<double> 
StandardHypoTestInvDemo(const char * infile = 0,
                        const char * wsName = "combined",
                        const char * modelSBName = "ModelConfig",
                        const char * modelBName = "",
                        const char * dataName = "obsData",                 
                        int calculatorType = 0,
                        int testStatType = 0, 
                        bool useCLs = true ,  
                        int npoints = 6,   
                        double poimin = 0,  
                        double poimax = 5, 
                        int ntoys=1000,
                        bool useNumberCounting = false,
                        const char * nuisPriorName = 0){
/*

  Other Parameter to pass in tutorial
  apart from standard for filename, ws, modelconfig and data

  type = 0 Freq calculator 
  type = 1 Hybrid calculator
  type = 2 Asymptotic calculator  
  type = 3 Asymptotic calculator using nominal Asimov data sets (not using fitted parameter values but nominal ones)

  testStatType = 0 LEP
  = 1 Tevatron 
  = 2 Profile Likelihood
  = 3 Profile Likelihood one sided (i.e. = 0 if mu < mu_hat)

  useCLs          scan for CLs (otherwise for CLs+b)    

  npoints:        number of points to scan , for autoscan set npoints = -1 

  poimin,poimax:  min/max value to scan in case of fixed scans 
  (if min >  max, try to find automatically)                           

  ntoys:         number of toys to use 

  useNumberCounting:  set to true when using number counting events 

  nuisPriorName:   name of prior for the nnuisance. This is often expressed as constraint term in the global model
  It is needed only when using the HybridCalculator (type=1)
  If not given by default the prior pdf from ModelConfig is used. 

  extra options are available as global paramwters of the macro. They major ones are: 

  plotHypoTestResult   plot result of tests at each point (TS distributions) (defauly is true)
  useProof             use Proof   (default is true) 
  writeResult          write result of scan (default is true)
  rebuild              rebuild scan for expected limits (require extra toys) (default is false)
  generateBinned       generate binned data sets for toys (default is false) - be careful not to activate with 
  a too large (>=3) number of observables 
  nToyRatio            ratio of S+B/B toys (default is 2)
    

*/

  vector<double> resnull;

   TString fileName(infile);
   if (fileName.IsNull()) { 
      fileName = "results/example_combined_GaussExample_model.root";
      std::cout << "Use standard file generated with HistFactory : " << fileName << std::endl;
   }
  
   // open file and check if input file exists
   TFile * file = TFile::Open(fileName); 
  
   // if input file was specified but not found, quit
   if(!file && !TString(infile).IsNull()){
      cout <<"file " << fileName << " not found" << endl;
      return resnull;
   } 
  
   // if default file not found, try to create it
   if(!file ){
      // Normally this would be run on the command line
      cout <<"will run standard hist2workspace example"<<endl;
      gROOT->ProcessLine(".! prepareHistFactory .");
      gROOT->ProcessLine(".! hist2workspace config/example.xml");
      cout <<"\n\n---------------------"<<endl;
      cout <<"Done creating example input"<<endl;
      cout <<"---------------------\n\n"<<endl;
    
      // now try to access the file again
      file = TFile::Open(fileName);
    
   }
  
   if(!file){
      // if it is still not there, then we can't continue
      cout << "Not able to run hist2workspace to create example input" <<endl;
      return resnull;
   }
  


   HypoTestInvTool calc;

   // set parameters
   calc.SetParameter("PlotHypoTestResult", plotHypoTestResult);
   calc.SetParameter("WriteResult", writeResult);
   calc.SetParameter("Optimize", optimize);
   calc.SetParameter("UseVectorStore", useVectorStore);
   calc.SetParameter("GenerateBinned", generateBinned);
   calc.SetParameter("NToysRatio", nToysRatio);
   calc.SetParameter("MaxPOI", maxPOI);
   calc.SetParameter("UseProof", useProof);
   calc.SetParameter("NWorkers", nworkers);
   calc.SetParameter("Rebuild", rebuild);
   calc.SetParameter("NToyToRebuild", nToyToRebuild);
   calc.SetParameter("MassValue", massValue.c_str());
   calc.SetParameter("MinimizerType", minimizerType.c_str());
   calc.SetParameter("PrintLevel", printLevel);
   calc.SetParameter("InitialFit",initialFit);
   calc.SetParameter("ResultFileName",resultFileName);
   calc.SetParameter("RandomSeed",randomSeed);


   RooWorkspace * w = dynamic_cast<RooWorkspace*>( file->Get(wsName) );
   HypoTestInverterResult * r = 0;  
   std::cout << w << "\t" << fileName << std::endl;
   if (w != NULL) {
      r = calc.RunInverter(w, modelSBName, modelBName,
                           dataName, calculatorType, testStatType, useCLs,
                           npoints, poimin, poimax,  
                           ntoys, useNumberCounting, nuisPriorName );    
      if (!r) { 
         std::cerr << "Error running the HypoTestInverter - Exit " << std::endl;
         return resnull;          
      }
   }
   else { 
      // case workspace is not present look for the inverter result
      std::cout << "Reading an HypoTestInverterResult with name " << wsName << " from file " << fileName << std::endl;
      r = dynamic_cast<HypoTestInverterResult*>( file->Get(wsName) ); //
      if (!r) { 
         std::cerr << "File " << fileName << " does not contain a workspace or an HypoTestInverterResult - Exit " 
                   << std::endl;
         file->ls();
         return resnull; 
      }
   }		
  
   vector<double> res =   calc.AnalyzeResult( r, calculatorType, testStatType, useCLs, npoints, infile );
  
   return res;
}



vector<double> 
RooStats::HypoTestInvTool::AnalyzeResult( HypoTestInverterResult * r,
                                          int calculatorType,
                                          int testStatType, 
                                          bool useCLs,  
                                          int npoints,
                                          const char * fileNameBase ){

   // analyize result produced by the inverter, optionally save it in a file 
   
   double upperLimit = r->UpperLimit();
   double ulError = r->UpperLimitEstimatedError();

   /*   
   std::cout << "The computed upper limit is: " << upperLimit << " +/- " << ulError << std::endl;
   // compute expected limit
   std::cout << " expected limit (median) " << r->GetExpectedUpperLimit(0) << std::endl;
   std::cout << " expected limit (-1 sig) " << r->GetExpectedUpperLimit(-1) << std::endl;
   std::cout << " expected limit (+1 sig) " << r->GetExpectedUpperLimit(1) << std::endl;
   std::cout << " expected limit (-2 sig) " << r->GetExpectedUpperLimit(-2) << std::endl;
   std::cout << " expected limit (+2 sig) " << r->GetExpectedUpperLimit(2) << std::endl;
   */

   // FOLLOWING ATLAS RECOMMENDATION [https://twiki.cern.ch/twiki/pub/AtlasProtected/StatisticsTools/Frequentist_Limit_Recommendation.pdf] 
  
   double CLs_obs = upperLimit;
   double CLs_exp = r->GetExpectedUpperLimit(0);
   int N=0; double sigma_s     = CLs_exp /(ROOT::Math::gaussian_quantile(1-(1-confidenceLevel)*ROOT::Math::gaussian_cdf(N,1),1) +N);
   N=+1;    double aCLs_plus1  = sigma_s *(ROOT::Math::gaussian_quantile(1-(1-confidenceLevel)*ROOT::Math::gaussian_cdf(N,1),1))+N*sigma_s;
   N=+2;    double aCLs_plus2  = sigma_s *(ROOT::Math::gaussian_quantile(1-(1-confidenceLevel)*ROOT::Math::gaussian_cdf(N,1),1))+N*sigma_s;
   N=-1;    double aCLs_minus1 = sigma_s *(ROOT::Math::gaussian_quantile(1-(1-confidenceLevel)*ROOT::Math::gaussian_cdf(N,1),1))+N*sigma_s;
   N=-2;    double aCLs_minus2 = sigma_s *(ROOT::Math::gaussian_quantile(1-(1-confidenceLevel)*ROOT::Math::gaussian_cdf(N,1),1))+N*sigma_s;
  //
  cout << Form("aCLs  [ obs/exp/+2/+1/-1/-2sigma UL at %2.0f % :: %20s at % 4.1f GeV ] = ",confidenceLevel*100, pname.Data(),xmass) 
       << Form(" %6.02f", CLs_obs) 
       << Form(" %6.02f", CLs_exp)
       << Form(" %6.02f", aCLs_plus2)
       << Form(" %6.02f", aCLs_plus1)
       << Form(" %6.02f", aCLs_minus1)
       << Form(" %6.02f", aCLs_minus2)
       << endl;

   // write result in a file 
   if (r != NULL && mWriteResult) {
    
      // write to a file the results
      const char *  calcType = (calculatorType == 0) ? "Freq" : (calculatorType == 1) ? "Hybr" : "Asym";
      const char *  limitType = (useCLs) ? "CLs" : "Cls+b";
      const char * scanType = (npoints < 0) ? "auto" : "grid";
      if (resultFileName.IsNull())
         resultFileName = TString::Format("%s_%s_%s_ts%d_",calcType,limitType,scanType,testStatType);      
      //strip the / from the filename
      if (mMassValue.size()>0) {
         resultFileName += mMassValue.c_str();
         resultFileName += "_";
      }
    
      TString name = fileNameBase; 
      name.Replace(0, name.Last('/')+1, "");
      resultFileName += name;
    
      TFile * fileOut = new TFile(resultFileName,"RECREATE");
      r->Write();
      fileOut->Close();                                                                     
   }   
  
  
   // plot the result ( p values vs scan points) 
   std::string typeName = "";
   if (calculatorType == 0 )
      typeName = "Frequentist";
   if (calculatorType == 1 )
      typeName = "Hybrid";   
   else if (calculatorType == 2 || calculatorType == 3) { 
      typeName = "Asymptotic";
      mPlotHypoTestResult = false; 
   }
  
   const char * resultName = r->GetName();
   TString plotTitle = TString::Format("%s CL Scan for workspace %s",typeName.c_str(),resultName);
   HypoTestInverterPlot *plot = new HypoTestInverterPlot("HTI_Result_Plot",plotTitle,r);

   // plot in a new canvas with style
   TString c1Name = TString::Format("%s_Scan",typeName.c_str());
   TCanvas * c1 = new TCanvas(c1Name); 
   c1->SetLogy(false);
   
   plot->Draw("CLb 2CL");  // plot all and Clb
  
   const int nEntries = r->ArraySize();
  
   // plot test statistics distributions for the two hypothesis 
   if (mPlotHypoTestResult) { 
      TCanvas * c2 = new TCanvas();
      if (nEntries > 1) { 
         int ny = TMath::CeilNint( sqrt(nEntries) );
         int nx = TMath::CeilNint(double(nEntries)/ny);
         c2->Divide( nx,ny);
      }
      for (int i=0; i<nEntries; i++) {
         if (nEntries > 1) c2->cd(i+1);
         SamplingDistPlot * pl = plot->MakeTestStatPlot(i);
         pl->SetLogYaxis(true);
         pl->Draw();
      }
   }

   // return the result
   vector<double> res;
   res.push_back(CLs_obs);
   res.push_back(CLs_exp);
   res.push_back(aCLs_plus2);
   res.push_back(aCLs_plus1);
   res.push_back(aCLs_minus1);
   res.push_back(aCLs_minus2);
   return res;
}



// internal routine to run the inverter
HypoTestInverterResult *
RooStats::HypoTestInvTool::RunInverter(RooWorkspace * w,
                                       const char * modelSBName, const char * modelBName, 
                                       const char * dataName, int type,  int testStatType, 
                                       bool useCLs, int npoints, double poimin, double poimax, 
                                       int ntoys,
                                       bool useNumberCounting,
                                       const char * nuisPriorName ){

   std::cout << "Running HypoTestInverter on the workspace " << w->GetName() << std::endl;
  
   //   w->Print();
  
  
   RooAbsData * data = w->data(dataName); 
   if (!data) { 
      Error("StandardHypoTestDemo","Not existing data %s",dataName);
      return 0;
   }
   else 
      std::cout << "Using data set " << dataName << std::endl;
  
   if (mUseVectorStore) { 
      RooAbsData::defaultStorageType = RooAbsData::Vector;
      data->convertToVectorStore() ;
   }
  
  
   // get models from WS
   // get the modelConfig out of the file
   ModelConfig* bModel = (ModelConfig*) w->obj(modelBName);
   ModelConfig* sbModel = (ModelConfig*) w->obj(modelSBName);
  
   if (!sbModel) {
      Error("StandardHypoTestDemo","Not existing ModelConfig %s",modelSBName);
      return 0;
   }
   // check the model 
   if (!sbModel->GetPdf()) { 
      Error("StandardHypoTestDemo","Model %s has no pdf ",modelSBName);
      return 0;
   }
   if (!sbModel->GetParametersOfInterest()) {
      Error("StandardHypoTestDemo","Model %s has no poi ",modelSBName);
      return 0;
   }
   if (!sbModel->GetObservables()) {
      Error("StandardHypoTestInvDemo","Model %s has no observables ",modelSBName);
      return 0;
   }
   if (!sbModel->GetSnapshot() ) { 
      Info("StandardHypoTestInvDemo","Model %s has no snapshot  - make one using model poi",modelSBName);
      sbModel->SetSnapshot( *sbModel->GetParametersOfInterest() );
   }
  
   // case of no systematics
   // remove nuisance parameters from model
   if (noSystematics) { 
      const RooArgSet * nuisPar = sbModel->GetNuisanceParameters();
      if (nuisPar && nuisPar->getSize() > 0) { 
         std::cout << "StandardHypoTestInvDemo" << "  -  Switch off all systematics by setting them constant to their initial values" << std::endl;
         RooStats::SetAllConstant(*nuisPar);
      }
      if (bModel) { 
         const RooArgSet * bnuisPar = bModel->GetNuisanceParameters();
         if (bnuisPar) 
            RooStats::SetAllConstant(*bnuisPar);
      }
   }
  
   if (!bModel || bModel == sbModel) {
      Info("StandardHypoTestInvDemo","The background model %s does not exist",modelBName);
      Info("StandardHypoTestInvDemo","Copy it from ModelConfig %s and set POI to zero",modelSBName);
      bModel = (ModelConfig*) sbModel->Clone();
      bModel->SetName(TString(modelSBName)+TString("_with_poi_0"));      
      RooRealVar * var = dynamic_cast<RooRealVar*>(bModel->GetParametersOfInterest()->first());
      if (!var) return 0;
      double oldval = var->getVal();
      var->setVal(0);
      bModel->SetSnapshot( RooArgSet(*var)  );
      var->setVal(oldval);
   }
   else { 
      if (!bModel->GetSnapshot() ) { 
         Info("StandardHypoTestInvDemo","Model %s has no snapshot  - make one using model poi and 0 values ",modelBName);
         RooRealVar * var = dynamic_cast<RooRealVar*>(bModel->GetParametersOfInterest()->first());
         if (var) { 
            double oldval = var->getVal();
            var->setVal(0);
            bModel->SetSnapshot( RooArgSet(*var)  );
            var->setVal(oldval);
         }
         else { 
            Error("StandardHypoTestInvDemo","Model %s has no valid poi",modelBName);
            return 0;
         }         
      }
   }

   // check model  has global observables when there are nuisance pdf
   // for the hybrid case the globobs are not needed
   if (type != 1 ) { 
      bool hasNuisParam = (sbModel->GetNuisanceParameters() && sbModel->GetNuisanceParameters()->getSize() > 0);
      bool hasGlobalObs = (sbModel->GetGlobalObservables() && sbModel->GetGlobalObservables()->getSize() > 0);
      if (hasNuisParam && !hasGlobalObs ) {  
         // try to see if model has nuisance parameters first 
         RooAbsPdf * constrPdf = RooStats::MakeNuisancePdf(*sbModel,"nuisanceConstraintPdf_sbmodel");
         if (constrPdf) { 
            Warning("StandardHypoTestInvDemo","Model %s has nuisance parameters but no global observables associated",sbModel->GetName());
            Warning("StandardHypoTestInvDemo","\tThe effect of the nuisance parameters will not be treated correctly ",sbModel->GetName());
         }
      }
   }


  
   // run first a data fit 
  
   const RooArgSet * poiSet = sbModel->GetParametersOfInterest();
   RooRealVar *poi = (RooRealVar*)poiSet->first();
  
   std::cout << "StandardHypoTestInvDemo : POI initial value:   " << poi->GetName() << " = " << poi->getVal()   << std::endl;  
  
   // fit the data first (need to use constraint )
   TStopwatch tw; 

   bool doFit = initialFit;
   if (testStatType == 0 && initialFit == -1) doFit = false;  // case of LEP test statistic
   if (type == 3  && initialFit == -1) doFit = false;         // case of Asymptoticcalculator with nominal Asimov
   double poihat = 0;
   if (doFit)  { 

      // do the fit : By doing a fit the POI snapshot (for S+B)  is set to the fit value
      // and the nuisance parameters nominal values will be set to the fit value. 
      // This is relevant when using LEP test statistics
     ROOT::Math::MinimizerOptions::SetDefaultTolerance(TOLERANCE);
      Info( "StandardHypoTestInvDemo"," Doing a first fit to the observed data ");
      if (minimizerType.size()==0) minimizerType = ROOT::Math::MinimizerOptions::DefaultMinimizerType();
      else 
         ROOT::Math::MinimizerOptions::SetDefaultMinimizer(minimizerType.c_str());
      Info("StandardHypoTestInvDemo","Using %s as minimizer for computing the test statistic",
           ROOT::Math::MinimizerOptions::DefaultMinimizerType().c_str() );
      RooArgSet constrainParams;
      if (sbModel->GetNuisanceParameters() ) constrainParams.add(*sbModel->GetNuisanceParameters());
      RooStats::RemoveConstantParameters(&constrainParams);
      tw.Start(); 
      RooFitResult * fitres = sbModel->GetPdf()->fitTo(*data,InitialHesse(false), Hesse(false),
                                                       Minimizer(minimizerType.c_str(),"Migrad"), Strategy(0), PrintLevel(mPrintLevel+1), Constrain(constrainParams), Save(true) );
      if (fitres->status() != 0) { 
         Warning("StandardHypoTestInvDemo","Fit to the model failed - try with strategy 1 and perform first an Hesse computation");
         fitres = sbModel->GetPdf()->fitTo(*data,InitialHesse(true), Hesse(false),Minimizer(minimizerType.c_str(),"Migrad"), Strategy(1), PrintLevel(mPrintLevel+1), Constrain(constrainParams), Save(true) );
      }
      if (fitres->status() != 0) { 
         Warning("StandardHypoTestInvDemo","Fit to the model failed - try with strategy 2 and perform first an Hesse computation");
         fitres = sbModel->GetPdf()->fitTo(*data,InitialHesse(true), Hesse(false),Minimizer(minimizerType.c_str(),"Migrad"), Strategy(2), PrintLevel(mPrintLevel+1), Constrain(constrainParams), Save(true) );
      }
      /*
      if (fitres->status() != 0) { 
         Warning("StandardHypoTestInvDemo","Fit to the model failed - Panic solution 1a: Try GSLMultiMin/bfgs2");

	 fitres = sbModel->GetPdf()->fitTo(*data,InitialHesse(false), Hesse(false),
					   Minimizer("GSLMultiMin","bfgs2"), Strategy(0), PrintLevel(mPrintLevel+1), 
					   Constrain(constrainParams), Save(true) );

      }
      if (fitres->status() != 0) { 
         Warning("StandardHypoTestInvDemo","Fit to the model failed - Panic solution 1b: Try GSLMultiMin/conjugatepr");

	 fitres = sbModel->GetPdf()->fitTo(*data,InitialHesse(false), Hesse(false),
					   Minimizer("GSLMultiMin","conjugatepr"), Strategy(0), PrintLevel(mPrintLevel+1), 
					   Constrain(constrainParams), Save(true) );

      }
      // GSLSimAn
      if (fitres->status() != 0) { 
         Warning("StandardHypoTestInvDemo","Fit to the model failed - Panic solution 1c: Try GSLSimAn");

	 fitres = sbModel->GetPdf()->fitTo(*data,InitialHesse(false), Hesse(false),
					   Minimizer("GSLSimAn"), Strategy(0), PrintLevel(mPrintLevel+1), 
					   Constrain(constrainParams), Save(true) );

      }
      */
      // --> if fit fails again, try to fit the mu=0 model first and start from these values ........................
      if (fitres->status() != 0) {
	Warning("StandardHypoTestInvDemo","Fit to the model failed - Panic solution 2: try to fit the mu=0 model first");
	Info( "StandardHypoTestInvDemo"," Doing a first fit to the observed data:  mu = 0 ");

        poi->setVal(0.0);
        poi->setConstant(1);
	
        RooFitResult * fitres_mu0 = sbModel->GetPdf()->fitTo
          (*data,InitialHesse(false), Hesse(false),
           Minimizer(minimizerType.c_str(),"Migrad"), Strategy(0), PrintLevel(mPrintLevel+1), Constrain(constrainParams), Save(true) );
	
        fitres_mu0->Print();
        std::cout << "FIT-mu=0 Status is: " << fitres_mu0->status() << endl;
        std::cout << " Fitted mu = " << poi->getVal() << " +- " <<  poi->getError() << std::endl;
        std::cout << "constrainParams:: " << constrainParams.getSize() << std::endl;
        constrainParams.Print("v");
	
        // revert to float mu 
        poi->setConstant(0);
	// do again the fit

	fitres = sbModel->GetPdf()->fitTo(*data,InitialHesse(false), Hesse(false),
					  Minimizer(minimizerType.c_str(),"Migrad"), Strategy(0), PrintLevel(mPrintLevel+1), Constrain(constrainParams), Save(true) );

      }

      if (fitres->status() != 0) {
	Warning("StandardHypoTestInvDemo","Fit to the model failed - Panic solution 3: try to fit the mu=1 model first");
	Info( "StandardHypoTestInvDemo"," Doing a first fit to the observed data:  mu = 1 ");

        poi->setVal(1.0);
        poi->setConstant(1);
	
        RooFitResult * fitres_mu1 = sbModel->GetPdf()->fitTo
          (*data,InitialHesse(false), Hesse(false),
           Minimizer(minimizerType.c_str(),"Migrad"), Strategy(0), PrintLevel(mPrintLevel+1), Constrain(constrainParams), Save(true) );
	
        fitres_mu1->Print();
        std::cout << "FIT-mu=0 Status is: " << fitres_mu1->status() << endl;
        std::cout << " Fitted mu = " << poi->getVal() << " +- " <<  poi->getError() << std::endl;
        std::cout << "constrainParams:: " << constrainParams.getSize() << std::endl;
        constrainParams.Print("v");
	
        // revert to float mu 
        poi->setConstant(0);
	// do again the fit

	fitres = sbModel->GetPdf()->fitTo(*data,InitialHesse(false), Hesse(false),
					  Minimizer(minimizerType.c_str(),"Migrad"), Strategy(0), PrintLevel(mPrintLevel+1), Constrain(constrainParams), Save(true) );

      }

      if (fitres->status() != 0) 
         Warning("StandardHypoTestInvDemo"," Fit still failed - continue anyway.....");
  
      std::cout << "FIT Status is: " << fitres->status() << endl;
      poihat  = poi->getVal();
      std::cout << "StandardHypoTestInvDemo - Best Fit value : " << poi->GetName() << " = "  
                << poihat << " +/- " << poi->getError() << std::endl;
      const RooArgSet* nuisPars = sbModel->GetNuisanceParameters();
      if (nuisPars) {cout << "nuisPars:: " << nuisPars->getSize() << endl;
	nuisPars->Print("v");}
      std::cout << "Time for fitting : "; tw.Print(); 
  
      //save best fit value in the poi snapshot 
      sbModel->SetSnapshot(*sbModel->GetParametersOfInterest());
      std::cout << "StandardHypoTestInvo: snapshot of S+B Model " << sbModel->GetName() 
                << " is set to the best fit value" << std::endl;
  
   }

   // print a message in case of LEP test statistics because it affects result by doing or not doing a fit 
   if (testStatType == 0) {
      if (!doFit) 
         Info("StandardHypoTestInvDemo","Using LEP test statistic - an initial fit is not done and the TS will use the nuisances at the model value");
      else 
         Info("StandardHypoTestInvDemo","Using LEP test statistic - an initial fit has been done and the TS will use the nuisances at the best fit value");
   }


   // build test statistics and hypotest calculators for running the inverter 
  
   SimpleLikelihoodRatioTestStat slrts(*sbModel->GetPdf(),*bModel->GetPdf());

   // null parameters must includes snapshot of poi plus the nuisance values 
   RooArgSet nullParams(*sbModel->GetSnapshot());
   if (sbModel->GetNuisanceParameters()) nullParams.add(*sbModel->GetNuisanceParameters());
   if (sbModel->GetSnapshot()) slrts.SetNullParameters(nullParams);
   RooArgSet altParams(*bModel->GetSnapshot());
   if (bModel->GetNuisanceParameters()) altParams.add(*bModel->GetNuisanceParameters());
   if (bModel->GetSnapshot()) slrts.SetAltParameters(altParams);
  
   // ratio of profile likelihood - need to pass snapshot for the alt
   RatioOfProfiledLikelihoodsTestStat 
      ropl(*sbModel->GetPdf(), *bModel->GetPdf(), bModel->GetSnapshot());
   ropl.SetSubtractMLE(false);
   ropl.SetPrintLevel(mPrintLevel);
   ropl.SetMinimizer(minimizerType.c_str());
  
   ProfileLikelihoodTestStat profll(*sbModel->GetPdf());
   if (testStatType == 3) profll.SetOneSided(1);
   profll.SetMinimizer(minimizerType.c_str());
   profll.SetPrintLevel(mPrintLevel);

   profll.SetReuseNLL(mOptimize);
   slrts.SetReuseNLL(mOptimize);
   ropl.SetReuseNLL(mOptimize);

   if (mOptimize) { 
      profll.SetStrategy(0);
      ropl.SetStrategy(0);
   }
  
   if (mMaxPoi > 0) poi->setMax(mMaxPoi);  // increase limit
  
   MaxLikelihoodEstimateTestStat maxll(*sbModel->GetPdf(),*poi); 
  
   // create the HypoTest calculator class 
   HypoTestCalculatorGeneric *  hc = 0;
   if (type == 0) hc = new FrequentistCalculator(*data, *bModel, *sbModel);
   else if (type == 1) hc = new HybridCalculator(*data, *bModel, *sbModel);
   else if (type == 2 ) hc = new AsymptoticCalculator(*data, *bModel, *sbModel);
   else if (type == 3 ) hc = new AsymptoticCalculator(*data, *bModel, *sbModel, true);  // for using Asimov data generated with nominal values 
   else {
      Error("StandardHypoTestInvDemo","Invalid - calculator type = %d supported values are only :\n\t\t\t 0 (Frequentist) , 1 (Hybrid) , 2 (Asymptotic) ",type);
      return 0;
   }
  
   // set the test statistic 
   TestStatistic * testStat = 0;
   if (testStatType == 0) testStat = &slrts;
   if (testStatType == 1) testStat = &ropl;
   if (testStatType == 2 || testStatType == 3) testStat = &profll;
   if (testStatType == 4) testStat = &maxll;
   if (testStat == 0) { 
      Error("StandardHypoTestInvDemo","Invalid - test statistic type = %d supported values are only :\n\t\t\t 0 (SLR) , 1 (Tevatron) , 2 (PLR), 3 (PLR1), 4(MLE)",testStatType);
      return 0;
   }
  
  
   ToyMCSampler *toymcs = (ToyMCSampler*)hc->GetTestStatSampler();
   if (toymcs) { 
      if (useNumberCounting) toymcs->SetNEventsPerToy(1);
      toymcs->SetTestStatistic(testStat);
    
      if (data->isWeighted() && !mGenerateBinned) { 
         Info("StandardHypoTestInvDemo","Data set is weighted, nentries = %d and sum of weights = %8.1f but toy generation is unbinned - it would be faster to set mGenerateBinned to true\n",data->numEntries(), data->sumEntries());
      }
      toymcs->SetGenerateBinned(mGenerateBinned);
  
      toymcs->SetUseMultiGen(mOptimize);
    
      if (mGenerateBinned &&  sbModel->GetObservables()->getSize() > 2) { 
         Warning("StandardHypoTestInvDemo","generate binned is activated but the number of ovservable is %d. Too much memory could be needed for allocating all the bins",sbModel->GetObservables()->getSize() );
      }

      // set the random seed if needed
      if (mRandomSeed >= 0) RooRandom::randomGenerator()->SetSeed(mRandomSeed); 
    
   }
  
  
   if (type == 1) { 
      HybridCalculator *hhc = dynamic_cast<HybridCalculator*> (hc);
      assert(hhc);
    
      hhc->SetToys(ntoys,ntoys/mNToysRatio); // can use less ntoys for b hypothesis 
    
      // remove global observables from ModelConfig (this is probably not needed anymore in 5.32)
      bModel->SetGlobalObservables(RooArgSet() );
      sbModel->SetGlobalObservables(RooArgSet() );
    
    
      // check for nuisance prior pdf in case of nuisance parameters 
      if (bModel->GetNuisanceParameters() || sbModel->GetNuisanceParameters() ) {

         // fix for using multigen (does not work in this case)
         toymcs->SetUseMultiGen(false);
         ToyMCSampler::SetAlwaysUseMultiGen(false);

         RooAbsPdf * nuisPdf = 0; 
         if (nuisPriorName) nuisPdf = w->pdf(nuisPriorName);
         // use prior defined first in bModel (then in SbModel)
         if (!nuisPdf)  { 
            Info("StandardHypoTestInvDemo","No nuisance pdf given for the HybridCalculator - try to deduce  pdf from the model");
            if (bModel->GetPdf() && bModel->GetObservables() ) 
               nuisPdf = RooStats::MakeNuisancePdf(*bModel,"nuisancePdf_bmodel");
            else 
               nuisPdf = RooStats::MakeNuisancePdf(*sbModel,"nuisancePdf_sbmodel");
         }   
         if (!nuisPdf ) {
            if (bModel->GetPriorPdf())  { 
               nuisPdf = bModel->GetPriorPdf();
               Info("StandardHypoTestInvDemo","No nuisance pdf given - try to use %s that is defined as a prior pdf in the B model",nuisPdf->GetName());            
            }
            else { 
               Error("StandardHypoTestInvDemo","Cannnot run Hybrid calculator because no prior on the nuisance parameter is specified or can be derived");
               return 0;
            }
         }
         assert(nuisPdf);
         Info("StandardHypoTestInvDemo","Using as nuisance Pdf ... " );
         nuisPdf->Print();
      
         const RooArgSet * nuisParams = (bModel->GetNuisanceParameters() ) ? bModel->GetNuisanceParameters() : sbModel->GetNuisanceParameters();
         RooArgSet * np = nuisPdf->getObservables(*nuisParams);
         if (np->getSize() == 0) { 
            Warning("StandardHypoTestInvDemo","Prior nuisance does not depend on nuisance parameters. They will be smeared in their full range");
         }
         delete np;
      
         hhc->ForcePriorNuisanceAlt(*nuisPdf);
         hhc->ForcePriorNuisanceNull(*nuisPdf);
      
      
      }
   } 
   else if (type == 2 || type == 3) { 
      if (testStatType == 3) ((AsymptoticCalculator*) hc)->SetOneSided(true);  
      if (testStatType != 2 && testStatType != 3)  
         Warning("StandardHypoTestInvDemo","Only the PL test statistic can be used with AsymptoticCalculator - use by default a two-sided PL");

      // ((AsymptoticCalculator*) hc)->SetQTilde(true); // not needed should be done automatically now
      ((AsymptoticCalculator*) hc)->SetPrintLevel(mPrintLevel+1); 
   }
   else if (type == 0 || type == 1) 
      ((FrequentistCalculator*) hc)->SetToys(ntoys,ntoys/mNToysRatio); 

  
   // Get the result
   RooMsgService::instance().getStream(1).removeTopic(RooFit::NumIntegration);
  
  
  
   HypoTestInverter calc(*hc);
   calc.SetConfidenceLevel(confidenceLevel);
  
  
   calc.UseCLs(useCLs);
   calc.SetVerbose(true);
  
   // can speed up using proof-lite
   if (mUseProof && mNWorkers > 1) { 
      ProofConfig pc(*w, mNWorkers, "", kFALSE);
      toymcs->SetProofConfig(&pc);    // enable proof
   }
  
  
   if (npoints > 0) {
      if (poimin > poimax) { 
         // if no min/max given scan between MLE and +4 sigma 
         poimin = int(poihat);
         poimax = int(poihat +  4 * poi->getError());
      }
      std::cout << "Doing a fixed scan  in interval : " << poimin << " , " << poimax << std::endl;
      calc.SetFixedScan(npoints,poimin,poimax);
   }
   else { 
      //poi->setMax(10*int( (poihat+ 10 *poi->getError() )/10 ) );
      std::cout << "Doing an  automatic scan  in interval : " << poi->getMin() << " , " << poi->getMax() << std::endl;
   }
  
   tw.Start();
   HypoTestInverterResult * r = calc.GetInterval();
   std::cout << "Time to perform limit scan \n";
   tw.Print();
  
   if (mRebuild) {
      calc.SetCloseProof(1);
      tw.Start();
      SamplingDistribution * limDist = calc.GetUpperLimitDistribution(true,mNToyToRebuild);
      std::cout << "Time to rebuild distributions " << std::endl;
      tw.Print();
    
      if (limDist) { 
         std::cout << "expected up limit " << limDist->InverseCDF(0.5) << " +/- " 
                   << limDist->InverseCDF(0.16) << "  " 
                   << limDist->InverseCDF(0.84) << "\n"; 
      
         //update r to a new updated result object containing the rebuilt expected p-values distributions
         // (it will not recompute the expected limit)
         if (r) delete r;  // need to delete previous object since GetInterval will return a cloned copy
         r = calc.GetInterval();
      
      }
      else 
         std::cout << "ERROR : failed to re-build distributions " << std::endl; 
   }
  
   return r;
}



void ReadResult(const char * fileName, const char * resultName="", bool useCLs=true) { 
   // read a previous stored result from a file given the result name

   StandardHypoTestInvDemo(fileName, resultName,"","","",0,0,useCLs);
}



// --> code by Nikos



void runMSSMHtautauLimits(vector<filesHolder> files,
			  TString workspace,
			  TString modelconfig,
			  TString dataname,
			  TString asimovData,
			  TString folder,
			  double CL, bool skipData, TString pName
			  ) 
{

  // some containers to store the produced values
  // for each mA, store the limit
  std::map<double, double> map_observed;
  std::map<double, double> map_median;
  std::map<double, double> map_medianP2sigma;
  std::map<double, double> map_medianP1sigma;
  std::map<double, double> map_medianM1sigma;
  std::map<double, double> map_medianM2sigma;

  // make the calculations for all the points here

  std::vector<filesHolder>::iterator it_mass = files.begin();

  for (; it_mass != files.end(); ++it_mass) {
    filesHolder currentFiles = (*it_mass);
    int nfiles = currentFiles.size(); // this is the number of tanbetas
    if (nfiles == 0) continue;
    double mass = (currentFiles[0]).mass;
     
    cout << "Running over mA= " << mass << endl;
    //
    // tan beta maps for given mA
    std::map<double, double> tb_observed;
    std::map<double, double> tb_median;
    std::map<double, double> tb_medianP2sigma;
    std::map<double, double> tb_medianP1sigma;
    std::map<double, double> tb_medianM1sigma;
    std::map<double, double> tb_medianM2sigma;
    //
    // count how many tan betas were actually tried and how many were successfull
    int nTanBetasOk_obs = 0;
    int nTanBetasOk_med = 0;
    int nTanBetasOk_mP2 = 0;
    int nTanBetasOk_mP1 = 0;
    int nTanBetasOk_mM1 = 0;
    int nTanBetasOk_mM2 = 0;

    int nTanBetas_obs = 0;
    int nTanBetas_med = 0;
    int nTanBetas_mP2 = 0;
    int nTanBetas_mP1 = 0;
    int nTanBetas_mM1 = 0;
    int nTanBetas_mM2 = 0;

    bool keepCount_obs = true;
    bool keepCount_med = true;
    bool keepCount_mP2 = true;
    bool keepCount_mP1 = true;
    bool keepCount_mM1 = true;
    bool keepCount_mM2 = true;

    double lastFailedTanbeta_obs = 0;
    double lastFailedTanbeta_med = 0;
    double lastFailedTanbeta_mP2 = 0;
    double lastFailedTanbeta_mP1 = 0;
    double lastFailedTanbeta_mM1 = 0;
    double lastFailedTanbeta_mM2 = 0;

    int  macro_counter = 0;

    std::vector<double> tanBetaList;
    int nTanbeta = currentFiles.size();

    filesHolder::iterator it_tanb = currentFiles.begin();
    for (; it_tanb != currentFiles.end(); ++it_tanb) {
      TString infile = (*it_tanb).fname;
      double  tanbeta= (*it_tanb).tanbeta;
      tanBetaList.push_back(tanbeta);
      TString tag = "mA"; tag+=int(mass); tag+="TB"; tag +=int(tanbeta);

      std::cout << "Working for mA=" << mass << " and tanb= " << tanbeta 
		<< " from file: " << infile << std::endl;
      //
      //      vector<double> result;
// 	result = runAsymptoticsCLs(infile.Data(),
// 				   workspace.Data(),
// 				   modelconfig.Data(), 
// 				   dataname.Data(),
// 				   asimovData.Data(),
// 				   "", "", 
// 				   string(folder.Data()),
// 				   string(tag.Data()), CL, skipData);
      //StandardHypoTestInvDemo(fname,wname,mname,bname,dname,2+blind,3,true,nscan,scanmin,scanmax,0,false,0); // AsymptoticCalculator

      double scanmin=0.0, scanmax = 6;
      int blind = 0;
      double scanstep=0.25;
      if (SCANSTEP>0) {
	// change the steps
	scanmin = SCANMIN;
	scanmax = SCANMAX;
	scanstep= SCANSTEP;
      }

      int    nscan   = (scanmax-scanmin)/scanstep + 1;
      vector<double> result = StandardHypoTestInvDemo(infile.Data(), //fname,
						      workspace.Data(), //wname,
						      modelconfig.Data(), //mname,
						      "", //bname,
						      dataname.Data(), //dname,
						      2+blind,
						      3,
						      true,
						      nscan,
						      scanmin,scanmax,0,false,0);
      //
      if (result.size() == 0) {
	cout << "ERROR!!!! something went wrong with the names..." << endl;
	abort();
      }
      // get the values
      double observed    = result.at(0);
      double median      = result.at(1);
      double medianP2sig = result.at(2);
      double medianP1sig = result.at(3);
      double medianM1sig = result.at(4);
      double medianM2sig = result.at(5);

      if(keepCount_obs) {++nTanBetas_obs;}
      if(keepCount_med) {++nTanBetas_med;}
      if(keepCount_mP2) {++nTanBetas_mP2;}
      if(keepCount_mP1) {++nTanBetas_mP1;}
      if(keepCount_mM1) {++nTanBetas_mM1;}
      if(keepCount_mM2) {++nTanBetas_mM2;}


      // if any of them is infinite then skip this step
      bool isInf_obs = checkInfinity("observed",    observed   );
      bool isInf_med = checkInfinity("median",      median     );
      bool isInf_mP2 = checkInfinity("medianP2sig", medianP2sig);
      bool isInf_mP1 = checkInfinity("medianP1sig", medianP1sig);
      bool isInf_mM2 = checkInfinity("medianM2sig", medianM2sig);
      bool isInf_mM1 = checkInfinity("medianM1sig", medianM1sig);

//       bool isInf_All = isInf_obs and isInf_med and isInf_mP2 and isInf_mP1 and isInf_mM2 and isInf_mM1;

//       if (isInf_All) {
//  	cout << "Infinity observed: will skip tan beta " << tanbeta << " for file " << infile << endl;
//  	lastFailedTanbeta = tanbeta;
//  	continue;
//       }
      // store them, if they are not infinite
      if ((not isInf_obs) and keepCount_obs) { tb_observed[tanbeta]      = observed;    ++nTanBetasOk_obs; }
      if ((not isInf_med) and keepCount_med) { tb_median[tanbeta]        = median;      ++nTanBetasOk_med; }
      if ((not isInf_mP2) and keepCount_mP2) { tb_medianP2sigma[tanbeta] = medianP2sig; ++nTanBetasOk_mP2; }
      if ((not isInf_mP1) and keepCount_mP1) { tb_medianP1sigma[tanbeta] = medianP1sig; ++nTanBetasOk_mP1; }
      if ((not isInf_mM1) and keepCount_mM1) { tb_medianM1sigma[tanbeta] = medianM1sig; ++nTanBetasOk_mM1; }
      if ((not isInf_mM2) and keepCount_mM2) { tb_medianM2sigma[tanbeta] = medianM2sig; ++nTanBetasOk_mM2; }

      if (keepCount_obs and isInf_obs) lastFailedTanbeta_obs = tanbeta;
      if (keepCount_med and isInf_med) lastFailedTanbeta_med = tanbeta;
      if (keepCount_mP2 and isInf_mP2) lastFailedTanbeta_mP2 = tanbeta;
      if (keepCount_mP1 and isInf_mP1) lastFailedTanbeta_mP1 = tanbeta;
      if (keepCount_mM1 and isInf_mM1) lastFailedTanbeta_mM1 = tanbeta;
      if (keepCount_mM2 and isInf_mM2) lastFailedTanbeta_mM2 = tanbeta;

      // stop counting if 
      if (observed < 1    and nTanBetas_obs>1) { keepCount_obs = false; }
      if (median   < 1    and nTanBetas_med>1) { keepCount_med = false; }
      if (medianP2sig < 1 and nTanBetas_mP2>1) { keepCount_mP2 = false; }
      if (medianP1sig < 1 and nTanBetas_mP1>1) { keepCount_mP1 = false; }
      if (medianM1sig < 1 and nTanBetas_mM1>1) { keepCount_mM1 = false; }
      if (medianM2sig < 1 and nTanBetas_mM2>1) { keepCount_mM2 = false; }

      // if you find result less than 1, then stop the loop
      if (observed < 1 and median < 1 and medianP2sig < 1 and medianP1sig < 1 
	  and medianM2sig < 1 and medianM1sig < 1) {
	cout << "Won't continue with other tan beta values: less than 1 limit obtained" << endl;
	break;
      }

    } // end loop over tanb
    //
    // Print the summary
    printDebugTableMA(mass, tb_observed, tb_median, tb_medianP2sigma, 
		      tb_medianP1sigma, tb_medianM1sigma, tb_medianM2sigma, tanBetaList);
    return;

    double lim_observed(0);
    double lim_median(0);
    double lim_medianP2sig(0);
    double lim_medianP1sig(0);
    double lim_medianM1sig(0);
    double lim_medianM2sig(0);

    if (nTanbeta == 1) {
      lim_observed    = tb_observed[20];
      lim_median      = tb_median[20];
      lim_medianP2sig = tb_medianP2sigma[20];
      lim_medianP1sig = tb_medianP1sigma[20];
      lim_medianM1sig = tb_medianM1sigma[20];
      lim_medianM2sig = tb_medianM2sigma[20];
    }
    else {
      //continue;
      // extrapolate to find the limit:
      // we have to find the tanb that corresponds to mu = 1 limit
      lim_observed = getTanBeta(tb_observed);
      lim_median   = getTanBeta(tb_median);
      
      //  double lim_medianP2sig = getExrapolationGivenTanb(lim_median, tb_medianP2sigma);
      //  double lim_medianP1sig = getExrapolationGivenTanb(lim_median, tb_medianP1sigma);
      //  double lim_medianM1sig = getExrapolationGivenTanb(lim_median, tb_medianM1sigma);
      //  double lim_medianM2sig = getExrapolationGivenTanb(lim_median, tb_medianM2sigma);
      
      lim_medianP2sig = getTanBeta(tb_medianP2sigma);
      lim_medianP1sig = getTanBeta(tb_medianP1sigma);
      lim_medianM1sig = getTanBeta(tb_medianM1sigma);
      lim_medianM2sig = getTanBeta(tb_medianM2sigma);
      
    }
    if (nTanBetas_obs != nTanBetasOk_obs) {printf("DEBUG-LIMWARNING obs: Run %2i tanb but only %2i successful. Last failed tanb=%4.1f\n", nTanBetas_obs, nTanBetasOk_obs, lastFailedTanbeta_obs);}
    if (nTanBetas_med != nTanBetasOk_med) {printf("DEBUG-LIMWARNING med: Run %2i tanb but only %2i successful. Last failed tanb=%4.1f\n", nTanBetas_med, nTanBetasOk_med, lastFailedTanbeta_med);}
    if (nTanBetas_mP2 != nTanBetasOk_mP2) {printf("DEBUG-LIMWARNING mP2: Run %2i tanb but only %2i successful. Last failed tanb=%4.1f\n", nTanBetas_mP2, nTanBetasOk_mP2, lastFailedTanbeta_mP2);}
    if (nTanBetas_mP1 != nTanBetasOk_mP1) {printf("DEBUG-LIMWARNING mP1: Run %2i tanb but only %2i successful. Last failed tanb=%4.1f\n", nTanBetas_mP1, nTanBetasOk_mP1, lastFailedTanbeta_mP1);}
    if (nTanBetas_mM1 != nTanBetasOk_mM1) {printf("DEBUG-LIMWARNING mM1: Run %2i tanb but only %2i successful. Last failed tanb=%4.1f\n", nTanBetas_mM1, nTanBetasOk_mM1, lastFailedTanbeta_mM1);}
    if (nTanBetas_mM2 != nTanBetasOk_mM2) {printf("DEBUG-LIMWARNING mM2: Run %2i tanb but only %2i successful. Last failed tanb=%4.1f\n", nTanBetas_mM2, nTanBetasOk_mM2, lastFailedTanbeta_mM2);}

    //if (nTanBetas_obs == 1) {printf("DEBUG-LIMWARNING obs: only 1 tanb point successful\n");}
    if (nTanBetas_med == 1) {printf("DEBUG-LIMWARNING med: only 1 tanb point successful\n");}
    if (nTanBetas_mP2 == 1) {printf("DEBUG-LIMWARNING mP2: only 1 tanb point successful\n");}
    if (nTanBetas_mP1 == 1) {printf("DEBUG-LIMWARNING mP1: only 1 tanb point successful\n");}
    if (nTanBetas_mM1 == 1) {printf("DEBUG-LIMWARNING mM1: only 1 tanb point successful\n");}
    if (nTanBetas_mM2 == 1) {printf("DEBUG-LIMWARNING mM2: only 1 tanb point successful\n");}

    printf("DEBUG-LIMITS   mA %4.0f   tanb  %6.2f %6.2f %6.2f %6.2f %6.2f %6.2f\n", 
	   mass, lim_observed, lim_median, lim_medianP2sig,
	   lim_medianP1sig, lim_medianM1sig, lim_medianM2sig);
    // print also a limit code snipet to facilitate addition
    printf("DEBUG-MACRO: g0->SetPoint(%2i, %5.0f, %7.3f);\n", macro_counter, mass, lim_median);
    printf("DEBUG-MACRO: g1->SetPoint(%2i, %5.0f, %7.3f);\n", macro_counter, mass, lim_median);
    printf("DEBUG-MACRO: g2->SetPoint(%2i, %5.0f, %7.3f);\n", macro_counter, mass, lim_median);
    printf("DEBUG-MACRO: g1->SetPointError(%2i, 0,0, %7.3f, %7.3f);\n", macro_counter, lim_median-lim_medianM1sig, lim_medianP1sig-lim_median);
    printf("DEBUG-MACRO: g2->SetPointError(%2i, 0,0, %7.3f, %7.3f);\n", macro_counter, lim_median-lim_medianM2sig, lim_medianP2sig-lim_median);
    ++macro_counter;

    map_observed[mass]      = lim_observed;
    map_median[mass]        = lim_median;
    map_medianP2sigma[mass] = lim_medianP2sig;
    map_medianP1sigma[mass] = lim_medianP1sig;
    map_medianM1sigma[mass] = lim_medianM1sig;
    map_medianM2sigma[mass] = lim_medianM2sig;

  } // end loop over masses

  return;

  printf("Limits summary: CLs@%3.2f \n", CL);
  printf("Mass        Observed   Expected     Bands:  Expected:1sigma    Expected: 2 sigma\n");
  std::map<double, double>::iterator it_map_median = map_median.begin();

  for (; it_map_median != map_median.end(); ++it_map_median) {
    double mass = (*it_map_median).first;
    double median =(*it_map_median).second;

    double medianP2sig = map_medianP2sigma[mass];
    double medianP1sig = map_medianP1sigma[mass];
    double medianM1sig = map_medianM1sigma[mass];
    double medianM2sig = map_medianM2sigma[mass];

    double observed    = map_observed[mass];
    if (observed != observed) observed = 0;
    printf("%5.1fGeV   %5.2f      %5.2f                 %5.2f - %5.2f       %5.2f - %5.2f\n",
	   mass, observed, median, medianM1sig, medianP1sig, medianM2sig, medianP2sig);
    
    
  }


  plotMaker(map_median, map_medianP2sigma, map_medianP1sigma, map_medianM1sigma, 
	    map_medianM2sigma, map_observed,	    false, pName);


    
}




void runMSSMHtautauLimitsConfigGeneral(TString pathPref, TString WhatSyst, TString pName, double Mmin, double Mmax) {
  //
  // -- Configuration set-up ------------------------
  //

  // Workspace names ....................................................
  TString workspace("combined");
  TString modelconfig("ModelConfig");
  TString dataname("obsData");
  TString asimovData("asimovData");

  double CL(0.95);

  TString folder("test");

  // Blind Analysis?
  bool  skipData= false;

  // FILE NAMES:
  TString extra("");
  if (not runMSSM) extra += ("Independent"+IndPref);

  TString pathName("/home/rompotis/Analysis/higgsTauTau/December2012Analysis/Limits/MSSMLimits/TestSensitivities/RunMSSMStyleLimits/VersionV3/FinalWed7March/UnblindData/Root32.01/JulianSwagatoCode/SwagatoSusggestions/LimitApril25/TrySeveralFitsBeforeV2/WithStandaloneCompiledRootVersion/TestFullChain_100GeV/"+RESULTSVERSION+"/results/"+pathPref);
  //TString pathName("/phys/groups/tev/scratch3/users/rompotis/CombinationResults/results"
  //		   +RESULTSVERSION+"/mssm_");
  //mssm_mA450tb30_comb_combined_AllSyst_model.root
  vector<filesHolder> files = makeVectorFilesHolder(pathName, WhatSyst, Mmin, Mmax);

  //
  // -- End Configuration set-up --------------------
  //

  runMSSMHtautauLimits(files, workspace, modelconfig, dataname, asimovData, 
		       folder, CL, skipData, pName);

}

#ifdef USE_AS_MAIN

int main(int argc, char* argv[]) {

  if (argc < 3) {
    cout << "At least 2 argument is needed: ./myExe <runOption> <FACTOR option>" << endl;
    cout << "Run option:  1, 2, 3 : b-tag, b-veto, combined all syst"   << endl;
    cout << "            11,21,31 : b-tag, b-veto, combined stats only" << endl;
    cout << "Factor option:   0:" << endl;
    return 1;
  }

  int  option = atoi(argv[1]);
  float  factorOption = atof(argv[2]);
  if (factorOption > 1) {
    FACTOR = double(factorOption);
  }
  else {
    FACTOR = 0;
  }


  
  // ...................................................

  // modification: Mmin, Mmax become now -> mass and tanbeta points

  double Mmin = 90.;
  double Mmax = 300.;

  if (argc>=5) {
    Mmin = double (  atoi(argv[3])  );
    Mmax = double (  atoi(argv[4])  );
    cout << "Warning: will run job for mass/tanbeta range: " << Mmin << " / " << Mmax << endl;
  }
  if (argc>=8) {
    SCANSTEP= double( atof(argv[5]) );
    SCANMIN = double( atof(argv[6]) );
    SCANMAX = double( atof(argv[7]) );
    if (SCANSTEP>0)
      cout << "Scanning is manually adjusted to step " << SCANSTEP << " min " << SCANMIN  << " max " << SCANMAX << endl;
  }

  if (argc>=9) {
    TOLERANCE =  double( atof(argv[8]) );
  }
  cout << "TOLERANCE is " << TOLERANCE << endl;

  cout << "Asymptotic Limits with step factor " << FACTOR << endl;
  TString suff;
  suff+=int(FACTOR*100);


  // some sets from swagato's code ......................
  maxPOI=100;
  //  ROOT::Math::MinimizerOptions::SetDefaultPrintLevel(-1); 
  ROOT::Math::MinimizerOptions::SetDefaultTolerance(TOLERANCE);
  ROOT::Math::MinimizerOptions::SetDefaultStrategy(0);
  ROOT::Math::MinimizerOptions::SetDefaultMinimizer("Minuit2");
  printLevel=0;
  writeResult=false;
  plotHypoTestResult=false;


  switch (option) {

  case 1:
    cout << "Running b-tag sample" << endl;
    runMSSMHtautauLimitsConfigGeneral("chbTaggedDef_",  "AllSyst",  "exp_Case1_bTaggedDef_AllSyst"+suff+".cc", Mmin, Mmax);
    break;
  case 2:
    cout << "Running b-veto sample" << endl;
    runMSSMHtautauLimitsConfigGeneral("chCompl_",       "AllSyst",  "exp_Case3_Compl_AllSyst"+suff+".cc", Mmin, Mmax);
    break;
  case 3:
    cout << "Running b-tag+b-veto sample" << endl;
    runMSSMHtautauLimitsConfigGeneral("bTagDefCom_",    "AllSyst",  "exp_Case6_COMBbTagCom_AllSyst"+suff+".cc", Mmin, Mmax);
    break;
    // some extra points: stats only
    
  case 11:
    cout << "Running b-tag sample, stat only" << endl;
    runMSSMHtautauLimitsConfigGeneral("chbTaggedDef_",  "StatOnly", "exp_Case1_bTaggedDef_StatOnly"+suff+".cc", Mmin, Mmax);
    break;
  case 21:
    cout << "Running b-veto sample, stat only" << endl;
    runMSSMHtautauLimitsConfigGeneral("chCompl_",       "StatOnly", "exp_Case3_Compl_StatOnly"+suff+".cc", Mmin, Mmax);
    break;
  case 31:
    cout << "Running b-tag+b-veto sample, stat only" << endl;
    runMSSMHtautauLimitsConfigGeneral("bTagDefCom_",    "StatOnly", "exp_Case6_COMBbTagCom_StatOnly"+suff+".cc", Mmin, Mmax);
    break;


    // Analysis with MET cut

  case 4:
    cout << "Running b-tag sample with MET>20GeV" << endl;
    runMSSMHtautauLimitsConfigGeneral("chbTaggedMET_",  "AllSyst",  "exp_Case2_bTaggedMET_AllSyst"+suff+".cc", Mmin, Mmax);
    break;

  case 5:
    cout << "Running b-veto sample with MET>20GeV" << endl;
    runMSSMHtautauLimitsConfigGeneral("chComplMET_",       "AllSyst",  "exp_Case4_ComplMET_AllSyst"+suff+".cc", Mmin, Mmax);
    break;

  case 6:
    cout << "Running b-tag+b-veto sample with MET>20GeV" << endl;
    runMSSMHtautauLimitsConfigGeneral("bTagMETComMET_",    "AllSyst",  "exp_Case8_COMBbTagComMET_AllSyst"+suff+".cc", Mmin, Mmax);
    break;


    // stat only

  case 41:
    cout << "Running b-tag sample with MET>20GeV, stat only" << endl;
    runMSSMHtautauLimitsConfigGeneral("chbTaggedMET_",  "StatOnly",  "exp_Case2_bTaggedMET_StatOnly"+suff+".cc", Mmin, Mmax);
    break;

  case 51:
    cout << "Running b-veto sample with MET>20GeV, stat only" << endl;
    runMSSMHtautauLimitsConfigGeneral("chComplMET_",    "StatOnly",  "exp_Case4_ComplMET_StatOnly"+suff+".cc", Mmin, Mmax);
    break;

  case 61:
    cout << "Running b-tag+b-veto sample with MET>20GeV, stat only" << endl;
    runMSSMHtautauLimitsConfigGeneral("bTagMETComMET_", "StatOnly",  "exp_Case8_COMBbTagComMET_StatOnly"+suff+".cc", Mmin, Mmax);
    break;


    // model independent limits for bbA
  case 7:
    RESULTSVERSION="V6";
    TAGMI="bbA";
    cout << "Running b-tag sample for " << TAGMI << endl;
    runMSSMHtautauLimitsConfigGeneral("chbTaggedDef_",  "AllSyst",  "exp_Case1_bTaggedDef_AllSyst"+suff+".cc", Mmin, Mmax);
    break;
  case 71:
    RESULTSVERSION="V6";
    TAGMI="ggA";
    cout << "Running b-tag sample for " << TAGMI << endl;
    runMSSMHtautauLimitsConfigGeneral("chbTaggedDef_",  "AllSyst",  "exp_Case1_bTaggedDef_AllSyst"+suff+".cc", Mmin, Mmax);
    break;

  case 8:
    RESULTSVERSION="V6";
    TAGMI="ggA";
    cout << "Running b-veto sample for" << TAGMI << endl;
    runMSSMHtautauLimitsConfigGeneral("chCompl_",       "AllSyst",  "exp_Case3_Compl_AllSyst"+suff+".cc", Mmin, Mmax);
    break;
  case 81:
    RESULTSVERSION="V6";
    TAGMI="bbA";
    cout << "Running b-veto sample for" << TAGMI << endl;
    runMSSMHtautauLimitsConfigGeneral("chCompl_",       "AllSyst",  "exp_Case3_Compl_AllSyst"+suff+".cc", Mmin, Mmax);
    break;


  case 9:
    RESULTSVERSION="V6";
    TAGMI="ggA";
    cout << "Running b-tag+b-veto sample for " << TAGMI  << endl;
    runMSSMHtautauLimitsConfigGeneral("chComb_",    "AllSyst",  "exp_Case6_COMBbTagCom_AllSyst"+suff+".cc", Mmin, Mmax);
    break;
  case 10:
    RESULTSVERSION="V6";
    TAGMI="bbA";
    cout << "Running b-tag+b-veto sample for " << TAGMI  << endl;
    runMSSMHtautauLimitsConfigGeneral("chComb_",    "AllSyst",  "exp_Case6_COMBbTagCom_AllSyst"+suff+".cc", Mmin, Mmax);
    break;  



  default:
    cout << "No such option" << endl;

  }

}


// Julian-Swagato's main code ------
/*
int main(int argc, char* argv[]){
  //
  gROOT->SetBatch(kTRUE);
  //
  RooMsgService::instance().setGlobalKillBelow(RooFit::FATAL); // lowest message level
  //
  //Argument variables
  //
  int    imode   = (argc>1) ? atoi(argv[1]) : 403;
  int    isys    = (argc>2) ? atoi(argv[2]) : 0;
  double mass    = (argc>3) ? atof(argv[3]) : 120.0;
  int    blind   = (argc>4) ? atoi(argv[4]) : 0;
  double scanmin = (argc>5) ? atof(argv[5]) : 0.0;
  double scanmax = (argc>6) ? atof(argv[6]) : 50.0;
  double scanstep= (argc>7) ? atof(argv[7]) : 0.25;
  //
  if (imode==400) pname = "combined";
  if (imode==401) pname = "chan0j";                 
  if (imode==402) pname = "chan1j";
  if (imode==403) pname = "chan2jVBF";            
  if (imode==404) pname = "chan2jVH";             
  if (imode==405) pname = "channelAllVBFHMET";    
  if (imode==406) pname = "channelEle0jetHMET";   
  if (imode==407) pname = "channelEle0jetLMET";   
  if (imode==408) pname = "channelEle1jetinHMET"; 
  if (imode==409) pname = "channelMuo0jetHMET";   
  if (imode==410) pname = "channelMuo0jetLMET";   
  if (imode==411) pname = "channelMuo1jetinHMET"; 
  if (imode==412) pname = "LLChannel";  
  if (imode==413) pname = "LHChannel";  
  if (imode==414) pname = "HHChannel";  
  //
  if (isys==0) sname="AllSYS";
  if (isys==1) sname="NoErr";
  if (isys==2) sname="NoStat";
  if (isys==3) sname="NoTheory";
  if (isys==4) sname="NoSYS";
  //
  wname = (imode==412||imode==413||imode==414) ? "combined" : pname;
  mname = "ModelConfig";
  bname = "";
  dname = "obsData";
  fname = Form("workspaces/updated/%s_%s_%3.0f.root",pname.Data(),sname.Data(),mass);
  xmass = mass;
  //
  TStopwatch timer;
  timer.Start();
  //
  // Guess based on previous results
  //
  if (imode==400) {scanmax=15;  scanmin = 0; scanstep=0.25;} // combined
  if (imode==401) {scanmax=100; scanmin = 0; scanstep=1.00;} // chan0j
  if (imode==402) {scanmax=50;  scanmin = 0; scanstep=0.50;} // chan1j
  if (imode==403) {scanmax=25;  scanmin = 0; scanstep=0.25;} // chan2jVBF
  if (imode==404) {scanmax=100; scanmin = 0; scanstep=1.00;} // chan2jVH
  if (imode==405) {scanmax=25;  scanmin = 0; scanstep=0.25;} // channelAllVBFHMET
  if (imode==406) {scanmax=100; scanmin = 0; scanstep=1.00;} // channelEle0jetHMET
  if (imode==407) {scanmax=200; scanmin = 0; scanstep=2.00;} // channelEle0jetLMET
  if (imode==408) {scanmax=25;  scanmin = 0; scanstep=0.25;} // channelEle1jetinHMET
  if (imode==409) {scanmax=100; scanmin = 0; scanstep=1.00;} // channelMuo0jetHMET
  if (imode==410) {scanmax=200; scanmin = 0; scanstep=2.00;} // channelMuo0jetLMET
  if (imode==411) {scanmax=25;  scanmin = 0; scanstep=0.25;} // channelMuo1jetinHMET
  if (imode==412) {scanmax=25;  scanmin = 0; scanstep=0.25;} // LLChannel
  if (imode==414) {scanmax=25;  scanmin = 0; scanstep=0.25;} // HHChannel
  if (imode==413) {// LHChannel
    if (mass<150) {
      scanmax=15;  scanmin = 0; scanstep=0.25;
    } else {
      scanmax=20;  scanmin = 0; scanstep=0.25;
    }
  } 
  //
  int    nscan   = (scanmax-scanmin)/scanstep + 1;
  maxPOI=100;
  ROOT::Math::MinimizerOptions::SetDefaultPrintLevel(-1);  
  printLevel=-1;
  writeResult=false;
  plotHypoTestResult=false;
  cout << "Reading workspace = " << wname << " modelconfig = " << mname << " and data = " << dname << " from file = " << fname << endl;
  StandardHypoTestInvDemo(fname,wname,mname,bname,dname,2+blind,3,true,nscan,scanmin,scanmax,0,false,0); // AsymptoticCalculator
  //
  timer.Stop(); 
  timer.Print();
  //
  return 0;
}
*/

#endif
