#include "DecayTable.h"
#include "Util.h"
#include <iostream>
#include <gsl/gsl_integration.h>

using namespace std;

// Particle naming
const char *dnames[4] = {" ","d ", "s ", "b "};
const char *unames[4] = {" ","u ", "c ", "t "};
const char *lnames[4] = {" ","e ", "mu", "ta"};
const char *nnames[4] = {" ","ve", "vm", "vt"};
const char *hnames[6] = {" ","h ", "H ", "A ", "H+", "H-"};
const char *vnames[5] = {" ","ga", "Z ", "W+", "W-"};
const int dPDG[4] = {0 ,1 , 3 , 5 };
const int uPDG[4] = {0 ,2 , 4 , 6 };
const int lPDG[4] = {0 ,11, 13, 15};
const int nPDG[4] = {0 ,12, 14, 16};
const int hPDG[5] = {0 ,25, 35, 36, 37};
const int vPDG[4] = {0 ,22, 23, 24};


DecayTable::DecayTable(THDM mod) {
  set_model(mod);
  qcd_on=true;
}


void DecayTable::set_model(THDM mod) {
  model = mod;
  sm = mod.get_SM();
  int i,j,k;
  for (i=1;i<5;i++) {
    gammatot_h[i]=-1.;
    gamma_hgg[i]=-1.;
    gamma_hgaga[i]=-1.;
    for (j=1;j<5;j++) {
      gamma_hvv[i][j]=-1.;
      for (k=1;k<5;k++) {
        gamma_uhd[i][j][k]=-1.;
        gamma_hdd[i][j][k]=-1.;
        gamma_huu[i][j][k]=-1.;
        gamma_hdu[i][j][k]=-1.;
        gamma_hll[i][j][k]=-1.;
        gamma_hln[i][j][k]=-1.;
        gamma_hvh[i][j][k]=-1.;
        gamma_hhh[i][j][k]=-1.;
      }
    }
  }
  
 
}


THDM DecayTable::get_model() {
  return model;
}


double DecayTable::get_gamma_uhd(int u, int h, int d) {

  if ((u<1)||(u>3)) return 0.;
  if (h!=4) return 0.;
  if ((d<1)||(d>3)) return 0.;

  if (gamma_uhd[u][h][d]>=0) return gamma_uhd[u][h][d];

  double M =  sm.get_umass_pole(u);
  double m1 = model.get_hmass(h);
  double m2 = sm.get_dmass_pole(d);
  
  if (M<(m1+m2)) {
    gamma_uhd[u][h][d]=0.;
    return gamma_uhd[u][h][d];
  }

  complex <double> cs,cp;
  model.get_coupling_hdu(h,d,u,cs,cp);
  cp = -cp;

  gamma_uhd[u][h][d] =  1./(16.*M_PI)*M*pow(1.-m1*m1/(M*M),2)*(pow(abs(cs),2)+pow(abs(cp),2));
  
  if (qcd_on) {
    double mt = sm.get_qmass_MSbar(6);
    double mb = sm.get_qmass_MSbar(5);
    double as = sm.run_alphas_MSbar(mt,mt,mb);

    double qH = pow(m1/M,2);
    
    double K = 1.+as/M_PI*(7.-8.*pow(M_PI,2)/9.-2.*log(1.-qH)+2.*(1.-qH)+(4./9.+2./3.*log(1.-qH))*pow(1.-qH,2));
    
    gamma_uhd[u][h][d] = gamma_uhd[u][h][d]*K;
  }

  return gamma_uhd[u][h][d];
}


double DecayTable::get_gamma_hdd(int h, int d1, int d2) {

  if ((h<1)||(h>4)) return 0.;
  if ((d1<1)||(d1>3)) return 0.;
  if ((d2<1)||(d2>3)) return 0.;

  if (gamma_hdd[h][d1][d2]>=0) return gamma_hdd[h][d1][d2];

  double M = model.get_hmass(h);
  double m1 = sm.get_dmass_pole(d1);
  double m2 = sm.get_dmass_pole(d2);

  if (M<(m1+m2)) {
    gamma_hdd[h][d1][d2] = 0.;
    return gamma_hdd[h][d1][d2];
  }

  complex <double> cs,cp;
  model.get_coupling_hdd(h,d1,d2,cs,cp);

  gamma_hdd[h][d1][d2] = hff_onshell(M,m1,m2,cs,cp,3,h);
  return gamma_hdd[h][d1][d2];
}


double DecayTable::get_gamma_huu(int h, int u1, int u2) {

  if ((h<1)||(h>4)) return 0.;
  if ((u1<1)||(u1>3)) return 0.;
  if ((u2<1)||(u2>3)) return 0.;

  if (gamma_huu[h][u1][u2]>=0) return gamma_huu[h][u1][u2];

  double M = model.get_hmass(h);
  double m1 = sm.get_umass_pole(u1);
  double m2 = sm.get_umass_pole(u2);

  if (M<(m1+m2)) {
    gamma_huu[h][u1][u2] = 0.;
    return gamma_huu[h][u1][u2];
  }

  complex <double> cs,cp;
  model.get_coupling_huu(h,u1,u2,cs,cp);

  gamma_huu[h][u1][u2] = hff_onshell(M,m1,m2,cs,cp,3,h);
  return gamma_huu[h][u1][u2];
}


double DecayTable::get_gamma_hdu(int h, int d, int u) {


  if ((h<1)||(h>4)) return 0.;
  if ((d<1)||(d>3)) return 0.;
  if ((u<1)||(u>3)) return 0.;

  if (gamma_hdu[h][d][u]>=0) return gamma_hdu[h][d][u];

  double M = model.get_hmass(h);
  double m1 = sm.get_dmass_pole(d);
  double m2 = sm.get_umass_pole(u);
  
  if (M<(m1+m2)) {
    gamma_hdu[h][d][u] = 0.;
    return gamma_hdu[h][d][u];
  }

  complex <double> cs,cp;
  model.get_coupling_hdu(h,d,u,cs,cp);

  gamma_hdu[h][d][u] = hff_onshell(M,m1,m2,cs,cp,3,h);
  return gamma_hdu[h][d][u];
}


double DecayTable::get_gamma_hll(int h, int l1, int l2) {

  if ((h<1)||(h>4)) return 0.;
  if ((l1<1)||(l1>3)) return 0.;
  if ((l2<1)||(l2>3)) return 0.;

  if (gamma_hll[h][l1][l2]>=0) return gamma_hll[h][l1][l2];

  double M = model.get_hmass(h);
  double m1 = sm.get_lmass_pole(l1);
  double m2 = sm.get_lmass_pole(l2);

  if (M<(m1+m2)) {
    gamma_hll[h][l1][l2] = 0.;
    return gamma_hll[h][l1][l2];
  }

  complex <double> cs,cp;
  model.get_coupling_hll(h,l1,l2,cs,cp);

  gamma_hll[h][l1][l2] = hff_onshell(M,m1,m2,cs,cp,1,h);

  return gamma_hll[h][l1][l2];
}


double DecayTable::get_gamma_hln(int h, int l, int n) {

  if ((h<1)||(h>4)) return 0.;
  if ((l<1)||(l>3)) return 0.;
  if ((n<1)||(n>3)) return 0.;

  if (gamma_hln[h][l][n]>=0) return gamma_hln[h][l][n];

  double M = model.get_hmass(h);
  double m1 = sm.get_lmass_pole(l);
  double m2 = 0.;

  if (M<(m1+m2)) {
    gamma_hln[h][l][n] = 0.;
    return gamma_hln[h][l][n];
  }

  complex <double> cs,cp;
  model.get_coupling_hln(h,l,n,cs,cp);

  
  gamma_hln[h][l][n] = hff_onshell(M,m1,m2,cs,cp,1,h);
  return gamma_hln[h][l][n];
}


double DecayTable::get_gamma_hgg(int h) {

  if (!qcd_on) return 0.;

  if ((h<1)||(h>3)) return 0.;

  if (gamma_hgg[h]>=0) return gamma_hgg[h];
  
  gamma_hgg[h] = hgg(h);
  return gamma_hgg[h];
}


double DecayTable::get_gamma_hgaga(int h) {

  if (!qcd_on) return 0.;

  if ((h<1)||(h>3)) return 0.;

  if (gamma_hgaga[h]>=0) return gamma_hgaga[h];
  
  gamma_hgaga[h] = hgaga(h);
  return gamma_hgaga[h];
}


double DecayTable::get_gamma_hvv(int h, int V) {

  if ((h<1)||(h>=4)) return 0.;
  if ((V<1)||(V>3)) return 0.;

  if ((h<=3)&&(V==1)) {
    return get_gamma_hgaga(h);
  }

  if (gamma_hvv[h][V]>=0) return gamma_hvv[h][V];
  
  double M = model.get_hmass(h);
  double m = sm.get_vmass(V);

  double GV = sm.get_gamma_V(V);
  gamma_hvv[h][V] = 0.;
  
  if (M<m+GV) {
    gamma_hvv[h][V] = 0.;
    return gamma_hvv[h][V];
  }

  if (M>(2.*m+GV)) {
    gamma_hvv[h][V] = hvv_onshell(h,V,M);
    return gamma_hvv[h][V];
  } else if (M>2.*m-GV) {
    // Interpolation between on-shell and off-shell
    gamma_hvv[h][V] = (2.-(M-(2.*m-GV))/(2*GV))*hvv_offshell(h,V,M);
    return gamma_hvv[h][V];
  } else if (M>m+2*GV) {
    // "Fully" off-shell
    gamma_hvv[h][V] = 2.*hvv_offshell(h,V,M);
    return gamma_hvv[h][V];
  }
  
  gamma_hvv[h][V] = 0.;
  return gamma_hvv[h][V];

}


double DecayTable::get_gamma_hvh(int H, int V, int h) {

  if ((H<1)||(H>4)||(h==H)) return 0.;
  if ((V<1)||(V>3)) return 0.;
  if ((h<1)||(h>4)) return 0.;
  
  if ((H==4)&&(V!=3)) return 0;
  if ((H!=4)&&(h==4)&&(V!=3)) return 0;
  
  if (V==1) return 0;

  if (gamma_hvh[H][V][h]>=0) return gamma_hvh[H][V][h];

  double M  = model.get_hmass(H);
  double m1  = model.get_hmass(h);
  double m2 = sm.get_vmass(V);
  
  double GV = sm.get_gamma_V(V);
  gamma_hvh[H][V][h] = 0.;
  
  if (M<=m1) {
    gamma_hvh[H][V][h] = 0.;
    return gamma_hvh[H][V][h];
  }

  if (M>(m1+m2+5*GV)) {
    gamma_hvh[H][V][h] = hvh_onshell(H,V,h,M);
    return gamma_hvh[H][V][h];
  } 
  
  if (M>m1+2*GV) {
    gsl_integration_workspace * w = gsl_integration_workspace_alloc(1000);
        
    double result, error;
    
    integration_params ip;
    ip.M = M;
    ip.m1 = m1;
    ip.m2 = m2;
    ip.gamma = GV;
  
    gsl_function F;
    F.function = &hvh_fcn;
    F.params = &ip;
      
    double k = pow(m1/M,2);
    double imin = 0.;
    double imax = 1.-k;
  
    gsl_error_handler_t *old_handler = gsl_set_error_handler_off();
    int status = gsl_integration_qags (&F,imin,imax,0,1e-5,1000,
			  w, &result, &error); 
    if (status) 
      if (status!=GSL_EROUND) {
        printf ("gsl error: %s\n", gsl_strerror (status));
        exit(-1);
      }
    gsl_set_error_handler(old_handler);
    old_handler = NULL;
    gsl_integration_workspace_free (w);
    w = NULL;
  
    complex <double> c;
    model.get_coupling_vhh(V,H,h,c);
  
    double dV = 0.;
    double stw = sm.get_sintw();
    double ctw = sm.get_costw();
    double GF = sm.get_GF();
    double MW = sm.get_MW();

    if (V==1) {
      gamma_hvh[H][V][h] = 0.;
      return gamma_hvh[H][V][h];
    } else if(V==2) {
      dV = 3./pow(ctw,2)*(7./12.-10./9.*pow(stw,2)+40./27.*pow(stw,4));
    } else if(V==3) {
      dV = 3.;
    } else {
      gamma_hvh[H][V][h] = 0.;
      return gamma_hvh[H][V][h];
    }

    double KHHV = 3.*GF/(16.*sqrt(2)*pow(M_PI,3))*pow(MW,2)*pow(abs(c),2)*M*dV;
    gamma_hvh[H][V][h] = KHHV*result;
    
    // Count both H+W- and H-W+ final states in partial width to charged states
    // also done in hvh_onshell
    if ((h==4)&&(V==3)) gamma_hvh[H][V][h] = 2*gamma_hvh[H][V][h];

    if (M>(m1+m2)) {
      double G2 = hvh_onshell(H,V,h,M); 
      gamma_hvh[H][V][h] = max(gamma_hvh[H][V][h],G2);
    }

    return gamma_hvh[H][V][h];
  }


  return gamma_hvh[H][V][h];
}


double DecayTable::get_gamma_hhh(int h, int h1, int h2) {

  if ((h<1)||(h>4)) return 0.;
  if ((h1<1)||(h1>4)) return 0.;
  if ((h2<1)||(h2>4)) return 0.;

  if (gamma_hhh[h][h1][h2]>=0) return gamma_hhh[h][h1][h2];
  double Sf = 1.;
  if ((h1==h2)&&(h1!=4)) Sf = 0.5;

  double M = model.get_hmass(h);
  double m1 = model.get_hmass(h1);
  double m2 = model.get_hmass(h2);

  complex <double> c;
  model.get_coupling_hhh(h,h1,h2,c);
  
  gamma_hhh[h][h1][h2] = 0.;

  if (M>(m1+m2)) {
    double M2    = pow(abs(c),2);
    gamma_hhh[h][h1][h2] = Sf/(8.*M_PI)*M2*PS2(M,m1,m2);
    return gamma_hhh[h][h1][h2];
  }

  return gamma_hhh[h][h1][h2];
}


double DecayTable::get_gammatot_top() {

  double gtot = 0.;
  
  gtot += sm.get_gamma_top();
  gtot += get_gamma_uhd(3,4,1);
  gtot += get_gamma_uhd(3,4,2);
  gtot += get_gamma_uhd(3,4,3);

  return gtot;

}


double DecayTable::get_gammatot_h(int h) {

  if (h>4) return 0.;

  if (gammatot_h[h]>=0) return gammatot_h[h];

  gammatot_h[h] = 0.;
  
  // Fermionic modes
  for (int i=1;i<4;i++) {
    for (int j=1;j<4;j++) {
       gammatot_h[h]+=get_gamma_hdd(h,i,j);
       gammatot_h[h]+=get_gamma_huu(h,i,j);
       gammatot_h[h]+=get_gamma_hdu(h,i,j);
       gammatot_h[h]+=get_gamma_hll(h,i,j);
       gammatot_h[h]+=get_gamma_hln(h,i,j);
    }
  }
  
  // Vector bosons
  for (int i=1;i<4;i++) {
     gammatot_h[h]+=get_gamma_hvv(h,i);
  }

  // Gluons
   gammatot_h[h] += get_gamma_hgg(h);

  // H -> VH
  for (int i=1;i<4;i++) {
    for (int j=1;j<=4;j++) {
       gammatot_h[h]+=get_gamma_hvh(h,i,j);
    }
  }

  // Higgses
  for (int i=1;i<5;i++) {
    for (int j=1;j<5;j++) {
       gammatot_h[h]+=get_gamma_hhh(h,i,j);
    }
  }

  return gammatot_h[h];
}


double  DecayTable::get_gammatot_v(int v) {
  return sm.get_gamma_V(v);
}



void DecayTable::print_decay_LesHouches(FILE* output, double br, int id1, int id2) {
  if (br>0) 
    fprintf(output,"     % 16.8e     %1i     %3i   %3i\n",br,2,id1,id2);
}


void DecayTable::print_decay(const char *h, const char *id1, const char *id2, double g, double br) { 
  if (br>0)
    printf("%2s -> %2s %2s %12.3e   %12.3e\n",h,id1,id2,g,br);
}


void DecayTable::print_top_decays() {
  double gtot = get_gammatot_top();
  double gt[4],br[4];

  gt[0] = sm.get_gamma_top();
  gt[1] = get_gamma_uhd(3,4,1);
  gt[2] = get_gamma_uhd(3,4,2);
  gt[3] = get_gamma_uhd(3,4,3);


  for (int i=0;i<4;i++) {
    br[i]=gt[i]/gtot;
  }

  printf("\nDecay table for %s\n", unames[3]);
  printf("Total width:%12.3e GeV      BR\n", gtot);
  print_decay(unames[3],vnames[3],dnames[3],gt[0],br[0]);
  print_decay(unames[3],hnames[4],dnames[1],gt[1],br[1]);
  print_decay(unames[3],hnames[4],dnames[2],gt[2],br[2]);
  print_decay(unames[3],hnames[4],dnames[3],gt[3],br[3]);
  printf("---------------------------------------\n");
}


void DecayTable::print_top_decays_LesHouches(FILE* output, bool full) {
  double gtot = get_gammatot_top();
  double gt[4],br[4];

  gt[0] = sm.get_gamma_top();
  gt[1] = get_gamma_uhd(3,4,1);
  gt[2] = get_gamma_uhd(3,4,2);
  gt[3] = get_gamma_uhd(3,4,3);


  for (int i=0;i<4;i++) {
    br[i]=gt[i]/gtot;
  }

  fprintf(output,"DECAY  6   % 16.8e   # top decays\n",gtot);
  if (full) {
    fprintf(output,"#            BR          NDA    ID1   ID2\n");
    print_decay_LesHouches(output,br[0],vPDG[3],dPDG[3]);
    print_decay_LesHouches(output,br[1],hPDG[4],dPDG[1]);
    print_decay_LesHouches(output,br[2],hPDG[4],dPDG[2]);
    print_decay_LesHouches(output,br[3],hPDG[4],dPDG[3]);
  }
}


void DecayTable::print_decays(int h) {
  print_decays(0,h,true,false);
}


void DecayTable::print_width(int h) {
  printf(" Total width for %s: %10.3e GeV\n", hnames[h], get_gammatot_h(h));
}


void DecayTable::print_decays_LesHouches(FILE* output, int h, bool full) {
  print_decays(output,h,full,true);
}


double DecayTable::br(double dG, double G) {

  double BR = 0.;

  if (G>0.) {
    BR = dG/G;
    if (BR<THDM::EPS) BR = 0.;
  }
  
  return BR;
}


void DecayTable::print_decays(FILE* output,int h, bool full, bool les) {

  if ((h<1)||(h>4)) return;

  double gtot = get_gammatot_h(h);

  if (les) {
    if (h==1) 
      fprintf(output,"DECAY  25   % 16.8e   # h1 decays, lightest CP-even Higgs\n",gtot);
    else if (h==2)
      fprintf(output,"DECAY  35   % 16.8e   # h2 decays, heaviest CP-even Higgs\n",gtot);
    else if (h==3)
      fprintf(output,"DECAY  36   % 16.8e   # h3 decays, CP-odd Higgs\n",gtot);
    else if (h==4)
      fprintf(output,"DECAY  37   % 16.8e   # Charged Higgs decays\n",gtot);
    fprintf(output,"#            BR          NDA    ID1   ID2\n");
  } else {
    printf("\nDecay table for %s\n", hnames[h]);
    printf("Total width:%12.3e GeV      BR\n", gtot);
  }

  if (!full) return;

  double gdd[4][4];
  double guu[4][4];
  double gdu[4][4];
  double gll[4][4];
  double gln[4][4];
  double gvv[4];
  double gvh[4][5];
  double ghh[5];
  double ghgg;
  double brdd[4][4];
  double bruu[4][4];
  double brdu[4][4];
  double brll[4][4];
  double brln[4][4];
  double brvv[4];
  double brvh[4][5];
  double brhh[5];
  double brhgg = 0.;
  
  // Fermion decay modes
  for (int i=1;i<4;i++) {
    for (int j=1;j<4;j++) {
      gdd[i][j]=get_gamma_hdd(h,i,j);
      guu[i][j]=get_gamma_huu(h,i,j);
      gdu[i][j]=get_gamma_hdu(h,i,j);
      gll[i][j]=get_gamma_hll(h,i,j);
      gln[i][j]=get_gamma_hln(h,i,j);
      brdd[i][j] = br(gdd[i][j],gtot);
      bruu[i][j] = br(guu[i][j],gtot);
      brdu[i][j] = br(gdu[i][j],gtot);
      brll[i][j] = br(gll[i][j],gtot);
      brln[i][j] = br(gln[i][j],gtot);
    }
  }

  // Vector bosons
  for (int i=1;i<4;i++) {
    gvv[i]=get_gamma_hvv(h,i);
    brvv[i]=br(gvv[i],gtot);
    for (int j=1;j<5;j++) {
      gvh[i][j]=get_gamma_hvh(h,i,j);
      brvh[i][j]=br(gvh[i][j],gtot);
    }
  }

  // Gluons
  ghgg = get_gamma_hgg(h);
  brhgg = br(ghgg,gtot);

  for (int i=1;i<=4;i++) {
    ghh[i]=get_gamma_hhh(h,i,i);
    brhh[i]=br(ghh[i],gtot);
  }

  if (h==4) {
    for (int j=1;j<4;j++) {
      for (int i=1;i<4;i++) {
        if (les)
	  print_decay_LesHouches(output,brdu[i][j],uPDG[j],-dPDG[i]);
	else 
	  print_decay(hnames[h],unames[j],dnames[i],gdu[i][j],brdu[i][j]);
      }
    }
    for (int i=1;i<4;i++) {
      for (int j=1;j<4;j++) {
        if (les)
	  print_decay_LesHouches(output,brln[i][j],-lPDG[i],nPDG[j]);
	else 
	  print_decay(hnames[h],lnames[i],nnames[j],gln[i][j],brln[i][j]);
      }
    }
    for (int i=2;i<4;i++) {
      for (int j=1;j<=4;j++) {
        int sgn = 1;
        if ((i==3)&&(h!=4)) sgn = -1;
        if (les)
	  print_decay_LesHouches(output,brvh[i][j],vPDG[i],sgn*hPDG[j]);
	else 
	  print_decay(hnames[h],vnames[i],hnames[j],gvh[i][j],brvh[i][j]);
      }
    }
  } else {
    for (int i=1;i<4;i++) {
      for (int j=1;j<4;j++) {
	if (les)
	  print_decay_LesHouches(output,brdd[i][j],dPDG[i],-dPDG[j]);
	else 
	  print_decay(hnames[h],dnames[i],dnames[j],gdd[i][j],brdd[i][j]);
	if (les)
	  print_decay_LesHouches(output,bruu[i][j],uPDG[i],-uPDG[j]);
	else 
	  print_decay(hnames[h],unames[i],unames[j],guu[i][j],bruu[i][j]);
      }
    }
    for (int i=1;i<4;i++) {
      for (int j=1;j<4;j++) {
	if (les)
	  print_decay_LesHouches(output,brll[i][j],lPDG[i],-lPDG[j]);
	else 
	  print_decay(hnames[h],lnames[i],lnames[j],gll[i][j],brll[i][j]);
      }    
    }
    for (int i=1;i<4;i++) {
      int sgn = 1;
      if (i==3) sgn = -1;
      if (les) {
	print_decay_LesHouches(output,brvv[i],vPDG[i],sgn*vPDG[i]);
      } else {
	print_decay(hnames[h],vnames[i],vnames[i+(sgn==-1)],gvv[i],brvv[i]);
      }
    }

    if (les) 
      print_decay_LesHouches(output,brhgg,21,21);
    else
      print_decay(hnames[h],"g ","g ",ghgg,brhgg);

    for (int i=1;i<=4;i++) {
      int sgn = 1;
      if (i==4) sgn = -1;
      if (les) {
        print_decay_LesHouches(output,brhh[i],hPDG[i],sgn*hPDG[i]);
      } else {
	print_decay(hnames[h],hnames[i],hnames[i+(sgn==-1)],ghh[i],brhh[i]);
      }
    }
    for (int i=1;i<4;i++) {
      for (int j=1;j<=3;j++) {
        if (les) {
	  print_decay_LesHouches(output,brvh[i][j],vPDG[i],hPDG[j]);
        } else 
	  print_decay(hnames[h],vnames[i],hnames[j],gvh[i][j],brvh[i][j]);
      }
    }

    // Special case with H+ W- charged conjugate final state
    if (les) {
      print_decay_LesHouches(output,0.5*brvh[3][4],vPDG[3],-hPDG[4]);
      print_decay_LesHouches(output,0.5*brvh[3][4],hPDG[4],-vPDG[3]);
    } else {
      print_decay(hnames[h],vnames[3],hnames[5],0.5*gvh[3][4],0.5*brvh[3][4]);
      print_decay(hnames[h],hnames[4],vnames[4],0.5*gvh[3][4],0.5*brvh[3][4]);
    }

  }
  
  if (!les) printf("---------------------------------------\n");
}

void DecayTable::set_qcd(bool set) {
  qcd_on=set;
}

double DecayTable::hvv_offshell(int h, int V,double M) {

  double m = sm.get_vmass(V);

  double GV = sm.get_gamma_V(V);
  double G = 0.;
 
  gsl_integration_workspace * w = gsl_integration_workspace_alloc (1000);
        
  double result, error;
    
  integration_params ip;
  ip.M = M;
  ip.m1 = m;
  ip.m2 = m;
  ip.gamma = GV;
  
  gsl_function F;
  F.function = &hvv_fcn;
  F.params = &ip;
      
  double k = pow(m/M,2);
  double imin = 0.;
  double imax = 1.-k;
  
  gsl_error_handler_t *old_handler = gsl_set_error_handler_off();
  int status = gsl_integration_qags (&F,imin,imax,0,1e-5,1000,
			w, &result, &error); 
    
  if (status) 
    if (status!=GSL_EROUND) {
      printf ("gsl error: %s\n", gsl_strerror (status));
      exit(-1);
    }
  gsl_set_error_handler(old_handler);
  gsl_integration_workspace_free (w);
  old_handler = NULL;
  w = NULL;

  complex <double> c;
  model.get_coupling_vvh(V,V,h,c);
  
  double dV = 0.;
  double stw = sm.get_sintw();
  double GF = sm.get_GF();

  if (V==1) {
    return 0.;
  } else if(V==2) {
    dV = 3.*(7./12.-10./9.*pow(stw,2)+40./27.*pow(stw,4));
  } else if(V==3) {
    dV = 3.;
  } else return 0.;

  double KHVV = 3.*pow(abs(c),2)*GF/(64.*sqrt(2.)*pow(M_PI,3))*M*dV;
  G = KHVV*result;


  return G;

}


double DecayTable::hvv_onshell(int h, int V, double M) {
  double m = sm.get_vmass(V);
  double Sf = 0.;

  if (V==1) {
    return 0.;
  } else if(V==2) {
    Sf = 0.5;
  } else if(V==3) {
    Sf = 1.;
  } else return 0.;
  
  if (M<2.*m) return 0.0;

  complex <double> c;
  model.get_coupling_vvh(V,V,h,c);

  double M2 = 0.;
  
  if (m>0) {
    double x = pow(M/m,2);
    M2    = 4.*pow(abs(c),2)*(3.-x+x*x/4.);
  } else {
    M2    = 16.*pow(abs(c),2);
  }

  double G = Sf/(32.*M_PI)*M2*PS2(M,m,m);

  return G;
}


double DecayTable::hvh_onshell(int H, int V, int h, double M) {
  double m1 = sm.get_vmass(V);
  double m2  = model.get_hmass(h);

  if (M<m1+m2) return 0.;

  complex <double> c;
  model.get_coupling_vhh(V,H,h,c);
  
  int dV = 1;

  if (V==1) {
    return 0.;
  } 
  if ((V==3) && (h==4)) {
    dV = 2;
  }
  

  double G = pow(abs(c),2)/(16.*M_PI)*pow(m1,2)/M*sqrt(L(pow(m2,2),pow(m1,2),pow(M,2)))*L(pow(m2,2),pow(M,2),pow(m1,2))*dV;

  return G;
}


double DecayTable::PS2(double M, double m1, double m2) {
  double absp1 = sqrt((pow(M,2)-pow(m1+m2,2))*(pow(M,2)-pow(m1-m2,2)))/(2.*M);
  double PS = absp1/pow(M,2);

  return PS;
}


double DecayTable::hff_onshell(double M, double m1, double m2, complex<double> cs, complex<double> cp, int Nc, int h) {

  double M2    = 2.*(pow(M,2)-pow(m1+m2,2))*pow(abs(cs),2)+
    2.*(pow(M,2)-pow(m1-m2,2))*pow(abs(cp),2);

  double G = Nc*1./(8.*M_PI)*M2*PS2(M,m1,m2);
  
  // Apply QCD corrections for decay to quarks
  if (Nc==3&&qcd_on) {
    if (h!=4) {
      int Nf = sm.get_Nactivef(M);
      double as = sm.run_alphas_MSbar(M,sm.get_qmass_MSbar(6),sm.get_qmass_MSbar(5))/M_PI;
      double K = 1.+5.67*as+(33.94-1.36*Nf)*pow(as,2);
      G = G*K;
    } else if (h==4) {
      double as = sm.run_alphas_MSbar(M,sm.get_qmass_MSbar(6),sm.get_qmass_MSbar(5))/M_PI;
      double K = 1.+5.67*as;
      G = G*K;
    }
  }

  return G;
}


double DecayTable::hgaga(int h) {

  complex <double> I(0.,1);

  double alpha 	= sm.get_alpha();
  double v  		= sm.get_v();
  double v2 		= sm.get_v2();
  double mW 		= sm.get_MW();
  
  double M 		= model.get_hmass(h);
  double M2 		= M*M;  

  double mt   	= sm.get_qmass_MSbar(6);
  double mb   	= sm.get_qmass_MSbar(5);  
  double mHp 		= model.get_hmass(4);

  double tau_W   = M2/(4.*mW*mW);
  double tau_Hp  = M2/(4.*mHp*mHp);

  int Nc = 3;

  complex <double> gS, gP;
  complex <double> S_sum(0.,0.), P_sum(0.,0.);

  // Down-type quarks
  for (int i=1;i<=3;i++) {
    model.get_coupling_hdd(h,i,i,gS,gP);
    double m = sm.get_dmass_MSbar(i);
    double mp = sm.get_dmass_pole(i);    

    if (m > 0) {
      double mrun = sm.run_qmass_MSbar(m,m,M,mt,mb);
      double tau = M2/(4.*mp*mp);
      complex <double> Sd = 2.*Nc*pow(-1./3,2)*gS*v/mrun*F_sf(tau);
      complex <double> Pd = 2.*Nc*pow(-1./3,2)*gP*v/mrun*F_pf(tau);
      S_sum = S_sum + Sd;
      P_sum = P_sum + Pd;
    }
  }

  // Up-type quarks
  for (int i=1;i<=3;i++) {
    model.get_coupling_huu(h,i,i,gS,gP);
    double m = sm.get_umass_MSbar(i);
    double mp = sm.get_umass_pole(i);        

    if (m > 0) {
      double mrun = sm.run_qmass_MSbar(m,m,M,mt,mb);
      double tau = M2/(4.*mp*mp);
      complex <double> Sd = 2.*Nc*pow(2./3,2)*gS*v/mrun*F_sf(tau);
      complex <double> Pd = 2.*Nc*pow(2./3,2)*gP*v/mrun*F_pf(tau);
      S_sum = S_sum + Sd;
      P_sum = P_sum + Pd;
    }
  }

  // Leptons
  for (int i=1;i<=3;i++) {
    model.get_coupling_hll(h,i,i,gS,gP);
    double m = sm.get_lmass_pole(i);
    
    if (m > 0) {
      double tau = M2/(4.*m*m);
      complex <double> Sd = 2.*pow(1.,2)*gS*v/m*F_sf(tau);
      complex <double> Pd = 2.*pow(1.,2)*gP*v/m*F_pf(tau);
      S_sum = S_sum + Sd;
      P_sum = P_sum + Pd;
    }
  }

  // Charged Higgs and W contribution to scalar operator
  complex <double> g_hww,g_hhchc;
  model.get_coupling_vvh(3,3,h,g_hww);
  model.get_coupling_hhh(h,4,4,g_hhchc);

  g_hww = g_hww*v/(2.*pow(mW,2));

  S_sum = S_sum + g_hww*F_1(tau_W) + g_hhchc/v*v2/(2.*pow(mHp,2))*F_0(tau_Hp);
  double G = pow(M,3)*pow(alpha,2)/(256.*pow(M_PI,3)*v2)*(pow(abs(S_sum),2)+pow(abs(P_sum),2));

  return G;
}


double DecayTable::hgg(int h) {

  double v  = sm.get_v();
  double v2 = sm.get_v2();
  
  double M = model.get_hmass(h);
  double M2 = M*M;  

  double mt   = sm.get_qmass_MSbar(6);
  double mb   = sm.get_qmass_MSbar(5);

  complex <double> S_sum(0.,0.);
  complex <double> P_sum(0.,0.);
  complex <double> gS,gP;

  // Down-type quarks
  for (int i=1;i<=3;i++) {
    model.get_coupling_hdd(h,i,i,gS,gP);
    double m = sm.get_dmass_MSbar(i);
    double mp = sm.get_dmass_pole(i);    

    if (m > 0) {
      double mrun = sm.run_qmass_MSbar(m,m,M,mt,mb);
      double tau = M2/(4.*mp*mp);
      complex <double> Sd = gS*v/mrun*F_sf(tau);
      complex <double> Pd = gP*v/mrun*F_pf(tau);
      S_sum = S_sum + Sd;
      P_sum = P_sum + Pd;
    }
  }

  // Up-type quarks
  for (int i=1;i<=3;i++) {
    model.get_coupling_huu(h,i,i,gS,gP);
    double m = sm.get_umass_MSbar(i);
    
    if (m > 0) {
      double mrun = sm.run_qmass_MSbar(m,m,M,mt,mb);
      double mp = sm.get_umass_pole(i);    
      double tau = M2/(4.*mp*mp);
      complex <double> Sd = gS*v/mrun*F_sf(tau);
      complex <double> Pd = gP*v/mrun*F_pf(tau);
      S_sum = S_sum + Sd;
      P_sum = P_sum + Pd;
    }
  }

  int Nf = sm.get_Nactivef(M);
  double as = sm.run_alphas_MSbar(M,sm.get_qmass_pole(6),sm.get_qmass_pole(5));

  double KS = 1;
  double KP = 1;
  // QCD corrections
  if (qcd_on) {
    KS = 1.+as/M_PI*(95./4.-7./6.*Nf);
    KP = 1.+as/M_PI*(97./4.-7./6.*Nf);
  }
  
  double G = pow(M,3)*pow(as,2)/(32.*pow(M_PI,3)*v2)*(KS*pow(abs(S_sum),2)+KP*pow(abs(P_sum),2));

  return G;
}


complex <double> DecayTable::F_sf(double t) {
  double ti = 1./t;
  complex <double> c;
  c = ti*(1.+(1.-ti)*ftau(t));
  return c;
}


complex <double> DecayTable::F_pf(double t) {
  double ti = 1./t;
  complex <double> c;
  c = ti*ftau(t);
  return c;
}


complex <double> DecayTable::F_0(double t) {
  double ti = 1./t;
  complex <double> c;
  c = ti*(-1.+ti*ftau(t));
  return c;
}


complex <double> DecayTable::F_1(double t) {
  double ti = 1./t;
  complex <double> c;
  c = 2.+3.*ti+3.*ti*(2.-ti)*ftau(t);
  return c;
}


complex <double> DecayTable::ftau(double t) {

  complex <double> c;
  complex <double> I(0.,1.);

  if (t<=1.) {
    double x = asin(sqrt(t));
    c = x*x;
  }

  if (t>1.) {
    complex <double> x = log((sqrt(t)+sqrt(t-1))/(sqrt(t)-sqrt(t-1)))-I*M_PI;
    c = -1./4.*x*x;
  }

  return c;
}


double DecayTable::L(double x, double y, double z) {
  double lam = pow(1.-x/z-y/z,2)-4.*x*y/pow(z,2);
  return lam;
}




